package ru.yandex.chemodan.app.docviewer.adapters.imagemagick;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.utils.ExecUtils2;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.commune.image.RotateAngle;
import ru.yandex.commune.image.imageMagick.FileCommand;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.commune.image.imageMagick.ImageMagickTargetInfo;
import ru.yandex.commune.image.imageMagick.resize.ResizeProportionalMinimizeOnly;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.file.File2;

/**
 * Converts image to web image with processing (rotation)
 *
 * @author ssytnik
 */
public class ImageMagickAdapter2 {
    private static final Pattern ORIG_FORMAT_PATTERN = Pattern.compile("orig-format:([a-zA-Z]+)", Pattern.MULTILINE);

    @Value("${imagemagick.max.image.edge}")
    private int maxImageEdge;
    @Value("${imagemagick.identify}")
    private String identify;

    @Autowired
    private ImageMagick imageMagick;

    // here, target image is sent to 'target' stream source, and temporary file is then deleted
    public ImageMagickConvertInfo convert(InputStreamSource source, OutputStreamSource target, Option<Integer> layer) {
        ImageMagickConvertInfo info = convert(source, layer);
        try {
            info.getTempFileO().get().readTo(target);
            return info.withoutTempFile();
        } finally {
            info.getTempFileO().get().deleteRecursiveQuietly();
        }
    }

    // in this scenario, client is responsible for deleting temp file specified in the result
    public ImageMagickConvertInfo convert(InputStreamSource source, Option<Integer> layer) {
        return IoUtils.withFile(source, sourceFile -> {
            File2 targetFile = FileUtils.createEmptyTempFile("im-output", ".tmp", sourceFile.parent());

            try {
                ImageMagickTargetInfo targetInfo = new ImageMagickTargetInfo(
                        new ResizeProportionalMinimizeOnly(Dimension.valueOf(maxImageEdge, maxImageEdge)),
                        targetFile.getName());

                if (layer.isPresent()) {
                    sourceFile = new File2(sourceFile.getAbsolutePath() + "[" + layer.get() + "]");
                }
                String report = imageMagick.generateThumbnails(
                        sourceFile, Cf.list(targetInfo), RotateAngle.AUTO, false, false);

                ImageFormat targetFormat = determineTargetFormat(report);
                Dimension targetSize = determineTargetSize(report, targetFile.getName());

                return new ImageMagickConvertInfo(Option.of(targetFile), targetFormat, targetSize);

            } catch (RuntimeException e) {
                // XXX actually need to delete file outside IoUtils.withFile, but, likely, it can't fall
                targetFile.deleteRecursiveQuietly();
                throw e;
            }
        });
    }

    // XXX ssytnik: works very slow for big plain text files
    // XXX ssytnik: works wrong for fb2: https://jira.yandex-team.ru/browse/DOCVIEWER-1313
    public boolean isSupported(InputStreamSource source) {
        return IoUtils.withFile(source, new IoFunction<File2, Boolean>() {
            @Override
            public Boolean applyWithException(File2 sourceFile) {
                if (isSupportedByIdentify(sourceFile)) {
                    return true;
                } else {
                    return FileCommand.getImageFormat(sourceFile).isPresent();
                }
            }

            private boolean isSupportedByIdentify(File2 sourceFile) {
                ListF<String> cmd = Cf.arrayList();
                cmd.add(identify);
                cmd.add(sourceFile.getAbsolutePath());

                ExecResult result = ExecUtils2.runScript(cmd);

                return result.getCode() == 0;
            }
        });
    }

    private ImageFormat determineTargetFormat(String report) {
        Matcher m = ORIG_FORMAT_PATTERN.matcher(report);
        if (m.find()) {
            return ImageMagick.determineThumbnailFormat(ImageFormat.valueOf2(m.group(1)));
        } else {
            throw new IllegalStateException("Unable to determine image format from report: " + report);
        }
    }

    private Dimension determineTargetSize(String report, String targetFilename) {
        Matcher m = Pattern.compile("\\b" + targetFilename + ":(\\d+x\\d+)", Pattern.MULTILINE).matcher(report);
        if (m.find()) {
            return Dimension.valueOf(m.group(1));
        } else {
            throw new IllegalStateException("Unable to determine target size from report: " + report);
        }
    }

}
