package ru.yandex.chemodan.app.docviewer.adapters.mongo;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.PreDestroy;

import com.mongodb.DB;
import com.mongodb.Mongo;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoCredential;
import com.mongodb.MongoException;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import org.jetbrains.annotations.NotNull;
import org.springframework.jmx.export.annotation.ManagedMetric;
import org.springframework.jmx.export.annotation.ManagedOperation;
import org.springframework.jmx.export.annotation.ManagedResource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@ManagedResource
public class MongoDbAdapter {

    private static final Logger logger = LoggerFactory.getLogger(MongoDbAdapter.class);
    private final Mongo mongo;
    private final String hosts;
    private final String db;

    public MongoDbAdapter(String user, String password, String hosts,
            String db, String dbWriteConcern, MongoClientOptions options)
    {
        this.db = db;
        this.hosts = hosts;
        this.mongo = createMongo(user, password, db, options, parseServerAddresses(hosts));

        getDatabase().setWriteConcern(WriteConcern.valueOf(dbWriteConcern));
    }

    private Mongo createMongo(String user, String password, String db,
            MongoClientOptions options, List<ServerAddress> serverAddresses)
    {
        if (StringUtils.isNotBlank(user)) {
            MongoCredential credential =
                    MongoCredential.createScramSha1Credential(user, db, password.toCharArray());
            return new MongoClient(serverAddresses, Cf.list(credential), options);
        } else {
            return new MongoClient(serverAddresses, options);
        }
    }

    @NotNull
    private List<ServerAddress> parseServerAddresses(String hosts) {
        List<ServerAddress> serverAddresses = new ArrayList<>();
        for (String host : StringUtils.split(hosts, ";,\n")) {
            serverAddresses.add(new ServerAddress(getByName(host)));
        }
        return serverAddresses;
    }

    private InetAddress getByName(String host) {
        try {
            return InetAddress.getByName(host);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @PreDestroy
    public void destroy() {
        mongo.close();
    }

    @ManagedOperation
    public void dropCollection(String name) {
        getDatabase().getCollection(name).drop();
    }

    @ManagedOperation
    public void dropDatabase() {
        getDatabase().dropDatabase();
    }

    public DB getDatabase() {
        final DB database = mongo.getDB(this.db);
        Check.notNull(database);
        return database;
    }

    @ManagedMetric()
    public String getDb() {
        return db;
    }

    @ManagedMetric()
    public String getHosts() {
        return hosts;
    }

    public Mongo getMongo() {
        return mongo;
    }

    @ManagedMetric()
    public boolean hasServerUp() {
        try {
            mongo.getDatabaseNames();
            return true;
        } catch (MongoException e) {
            logger.warn(e, e);
            return false;
        }
    }

}
