package ru.yandex.chemodan.app.docviewer.adapters.mongo;

import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.ReadPreference;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author akirakozov
 * @author ssytnik
 * @author vlsergey
 */
public class MongoDbUtils {

    private static final Logger logger = LoggerFactory.getLogger(MongoDbUtils.class);

    private static final String COLUMN_ID = "_id";

    private static final int MAX_BATCH_SIZE = 200;
    private static final int MAX_BACTH_CYCLES = 1_000_000;

    public static void closeQuietly(DBCursor dbCursor) {
        try {
            dbCursor.close();
        } catch (Throwable exc) {
            logger.error("Unable to close DBCursor: " + exc, exc);
        }
    }

    public static boolean findExists(DBCollection dbCollection, DBObject query) {
        return dbCollection.findOne(query) != null;
    }

    public static <T> Option<T> findOne(DBCollection dbCollection, DBObject query, Function<DBObject, T> function) {
        return Option.ofNullable(dbCollection.findOne(query)).map(function);
    }

    public static <T> Option<T> findOne(DBCollection dbCollection, DBObject query, ReadPreference preference, Function<DBObject, T> function) {
        return Option.ofNullable(dbCollection.findOne(query, null, null, preference)).map(function);
    }

    public static <T> Option<T> findOneSorted(
            DBCollection dbCollection, DBObject query, DBObject orderBy, Function<DBObject, T> function)
    {
        return findSortedLimited(dbCollection, query, Option.of(orderBy), Option.of(1), function).firstO();
    }

    public static <T> ListF<T> find(DBCollection dbCollection, DBObject query, Function<DBObject, T> function) {
        return findSortedLimited(dbCollection, query, Option.empty(), Option.empty(), function);
    }

    public static <T> ListF<T> findSortedLimited(
            DBCollection dbCollection, DBObject query,
            Option<DBObject> orderBy, Option<Integer> limit, Function<DBObject, T> function)
    {
        DBCursor dbCursor = dbCollection.find(query);
        if (orderBy.isPresent()) {
            dbCursor = dbCursor.sort(orderBy.get());
        }
        if (limit.isPresent()) {
            dbCursor = dbCursor.limit(limit.get());
        }

        try {
            return Cf.wrap(dbCursor).map(function).toList();
        } finally {
            closeQuietly(dbCursor);
        }
    }

    public static <T> ListF<T> findValues(DBCollection dbCollection, DBObject query,
            String columnName)
    {
        DBObject fieldsToSelect = new BasicDBObject();
        if (!StringUtils.equals(COLUMN_ID, columnName)) {
            fieldsToSelect.put(COLUMN_ID, 0);
        }
        fieldsToSelect.put(columnName, 1);

        DBCursor dbCursor = dbCollection.find(query, fieldsToSelect);
        try {
            IteratorF<DBObject> objectsIterator = Cf.wrap(dbCursor);
            IteratorF<T> keyIterator = objectsIterator.map(obj -> (T) obj.get(columnName));
            return keyIterator.toList();
        } finally {
            closeQuietly(dbCursor);
        }
    }

    public static void forEachBatch(DBCollection dbCollection, DBObject query, String indexField, int nThreads,
            Function1V<DBObject> callbackHandler)
    {
        try (MultithreadedBatchCallback<DBObject> batchCallbackHandler = new MultithreadedBatchCallback<>(callbackHandler, nThreads)) {
            forEachBatchInner(dbCollection, query, indexField, batchCallbackHandler);
        }
    }

    private static void forEachBatchInner(
            DBCollection dbCollection, DBObject mainCondition, String indexField,
            MultithreadedBatchCallback<DBObject> batchCallbackHandler)
    {
        ListF<DBObject> list;

        DBObject order = new BasicDBObject(indexField, 1);
        Option<Object> offset = Option.empty();

        int cycles = 0;
        do {
            DBObject query;
            if (offset.isPresent()) {
                DBObject offsetCondition = new BasicDBObject(indexField, new BasicDBObject("$gt", offset.get()));
                BasicDBList conditions = new BasicDBList();
                conditions.addAll(Cf.list(mainCondition, offsetCondition));
                query = new BasicDBObject("$and", conditions);
            } else {
                query = mainCondition;
            }

            list = RetryUtils.retry(3, () -> findSortedLimited(dbCollection, query,
                    Option.of(order), Option.of(MAX_BATCH_SIZE), Function.identityF()));
            batchCallbackHandler.execute(list);
            offset = list.lastO().map(dbObject -> dbObject.get(indexField));
            cycles++;
        } while (list.isNotEmpty() && cycles < MAX_BACTH_CYCLES);

        if (list.isNotEmpty()) {
            logger.error("forEachBatch seems infinite, review the code");
        }
    }

    public static void forEach(DBCollection dbCollection,
            Function1V<? super DBObject> callbackHandler)
    {
        DBCursor dbCursor = dbCollection.find();
        try {
            forEach(dbCursor, callbackHandler);
        } finally {
            closeQuietly(dbCursor);
        }
    }

    private static void forEach(DBCursor dbCursor, Function1V<? super DBObject> callbackHandler) {
        Cf.wrap(dbCursor).forEachRemaining(callbackHandler);
    }

    public static Option<Boolean> getBoolean(DBObject dbObject, String fieldName) {
        return getBoolean(dbObject.get(fieldName));
    }

    private static Option<Boolean> getBoolean(Object obj) {
        if (obj == null) {
            return Option.empty();
        }
        if (obj instanceof Boolean) {
            return Option.of((Boolean) obj);
        }
        if (obj instanceof Number) {
            return Option.of(((Number) obj).intValue() != 0);
        }
        return Option.of(Boolean.valueOf(String.valueOf(obj)));
    }

    public static <T extends Enum<T>> Option<T> getEnum(DBObject dbObject, String fieldName,
            EnumResolver<T> enumResolver)
    {
        Object object = dbObject.get(fieldName);
        if (object == null) {
            return Option.empty();
        }
        String strValue = String.valueOf(object);
        return enumResolver.valueOfO(strValue);
    }

    public static <S, T extends Enum<T>> Option<T> getEnum(S obj, EnumResolver<T> enumResolver) {
        if (obj == null) {
            return Option.empty();
        }
        String strValue = String.valueOf(obj);
        return enumResolver.valueOfO(strValue);
    }

    public static Option<Float> getFloat(DBObject dbObject, String fieldName) {
        return getFloat(dbObject.get(fieldName));
    }

    public static Option<Float> getFloat(Object obj) {
        if (obj == null) {
            return Option.empty();
        }

        try {
            if (obj instanceof Number) {
                return Option.of(((Number) obj).floatValue());
            }
            return Option.of(Float.valueOf(String.valueOf(obj)));
        } catch (NumberFormatException exc) {
            return Option.empty();
        }
    }

    public static Option<Instant> getInstant(DBObject dbObject, String field) {
        if (dbObject == null) {
            return Option.empty();
        }

        return getInstant(dbObject.get(field));
    }

    private static Option<Instant> getInstant(Object value) {
        if (value == null) {
            return Option.empty();
        }
        if (value instanceof Date) {
            return Option.of(new Instant(value));
        }
        if (value instanceof Instant) {
            return Option.of((Instant) value);
        }
        if (value instanceof Number) {
            return Option.of(new Instant(((Number) value).longValue()));
        }
        return Option.of(new Instant(Long.parseLong(String.valueOf(value))));
    }

    public static Option<Integer> getInteger(DBObject dbObject, String fieldName) {
        return getInteger(dbObject.get(fieldName));
    }

    private static Option<Integer> getInteger(Object obj) {
        if (obj == null) {
            return Option.empty();
        }
        try {
            if (obj instanceof Number) {
                return Option.of(((Number) obj).intValue());
            }
            return Option.of(Integer.valueOf(String.valueOf(obj)));
        } catch (NumberFormatException exc) {
            return Option.empty();
        }
    }

    public static Option<Long> getLong(DBObject dbObject, String fieldName) {
        return getLong(dbObject.get(fieldName));
    }

    private static Option<Long> getLong(Object obj) {
        if (obj == null) {
            return Option.empty();
        }
        try {
            if (obj instanceof Number) {
                return Option.of(((Number) obj).longValue());
            }
            return Option.of(Long.valueOf(String.valueOf(obj)));
        } catch (NumberFormatException exc) {
            return Option.empty();
        }
    }

    public static Map<String, Object> getMap(DBObject dbObject, String fieldName) {
        return getMap(
                dbObject, fieldName,
                Function.<String>identityF().andThen(Option.notNullF()),
                val -> Option.ofNullable(getValue(val)));
    }

    @SuppressWarnings("unchecked")
    public static <K, V> Map<K, V> getMap(DBObject dbObject, String fieldName,
            Function<String, Option<K>> keyDeserializer,
            Function<Object, Option<V>> valueDeserializer)
    {
        Object object = dbObject.get(fieldName);
        if (object == null) {
            return Collections.emptyMap();
        }
        Map<K, V> result = new LinkedHashMap<>();

        if (object instanceof DBObject) {

            DBObject dbObjectValue = (DBObject) object;
            Map<String, Object> map = dbObjectValue.toMap();
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                final Option<K> keyO = keyDeserializer.apply(entry.getKey());
                final Option<V> valueO = valueDeserializer.apply(entry.getValue());

                if (keyO.isEmpty()) {
                    logger.warn("Unable to deserialize key '{}' of map '{}' of object '{}'",
                            entry.getKey(), fieldName, dbObject);
                    continue;
                }
                if (valueO.isEmpty()) {
                    logger.warn(
                            "Unable to deserialize value '{}' of key '{}' of map '{}' of object '{}'",
                            entry.getValue(), keyO.get(), fieldName, dbObject);
                    continue;
                }

                final K key = keyO.get();
                final V value = valueO.get();

                result.put(key, value);
            }

        } else {
            logger.warn("Value of field '{}' of object '{}' is not a DBObject (not a map): {}",
                    fieldName, dbObject, object);
        }

        if (result.isEmpty()) {
            return Collections.emptyMap();
        }

        if (result.size() == 1) {
            Map.Entry<K, V> entry = result.entrySet().iterator().next();
            return Collections.singletonMap(entry.getKey(), entry.getValue());
        }

        return Collections.unmodifiableMap(result);
    }

    public static Option<String> getString(DBObject dbObject, String fieldName, boolean trim,
            boolean allowEmpties)
    {
        Object value = dbObject.get(fieldName);
        if (value == null) {
            return Option.empty();
        }
        String valueString = value instanceof String ? (String) value : String.valueOf(value);
        if (trim) {
            valueString = StringUtils.trimToEmpty(valueString);
        }
        if (!allowEmpties && StringUtils.isEmpty(valueString)) {
            return Option.empty();
        }
        return Option.of(valueString);
    }

    public static Option<String> getStringWithEmpties(DBObject dbObject, String fieldName) {
        return getString(dbObject, fieldName, false, true);
    }

    public static Option<String> getStringNoEmpties(DBObject dbObject, String fieldName) {
        return getString(dbObject, fieldName, false, false);
    }

    @SuppressWarnings("unchecked")
    public static <T> Function<DBObject, T> getValueF(final String columnName) {
        return dbObject -> (T) dbObject.get(columnName);
    }

    public static String calculateMd5IdKey(String id) {
        return Md5.A.digest(id).hex();
    }

    public static String calculateMd5IdKeyByActualUri(ActualUri uri) {
        return calculateMd5IdKey(uri.getUriString());
    }

    public static DBObject findRecordById(DBCollection collection, String id) {
        DBObject query = new BasicDBObject();
        query.put(COLUMN_ID, id);

        return collection.findOne(query);
    }

    public static void removeRecordById(DBCollection collection, String id) {
        DBObject query = new BasicDBObject();
        query.put(COLUMN_ID, id);

        collection.remove(query);
    }

    private static Object getValue(Object obj) {
        Option<Float> valF = getFloat(obj);
        if (valF.isPresent()) {
            return valF.get();
        }

        Option<Instant> valIns = getInstant(obj);
        if (valIns.isPresent()) {
            return valIns.get();
        }

        if (obj instanceof String) {
            return obj;
        }
        return null;
    }

}
