package ru.yandex.chemodan.app.docviewer.adapters.mongo;

import java.io.Closeable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.MoreExecutors;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Sequentially, one-by-one, handles task object batches.
 * A single batch is a list of task objects; they are consumed
 * to a given callback, using specified number of threads.
 * Batch execution blocks until its all tasks are finished.
 *
 * @author ssytnik
 */
public class MultithreadedBatchCallback<T> implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(MultithreadedBatchCallback.class);

    private final Function1V<T> callback;
    private final ExecutorService executor;

    /**
     * @param callback accepts task object, should be thread-safe
     * @param nThreads 0 to execute tasks in this thread, or number of threads in executor
     */
    public MultithreadedBatchCallback(Function1V<T> callback, int nThreads) {
        this.callback = callback;
        this.executor = nThreads == 0 ?
                MoreExecutors.newDirectExecutorService() :
                new ThreadPoolExecutor(nThreads, nThreads,
                        0L, TimeUnit.MILLISECONDS,
                        new LinkedBlockingQueue<>(nThreads));
    }

    @Override
    public void close() {
        executor.shutdownNow();
    }

    public void execute(ListF<T> tasks) {
        try {
            tasks.map((Function<T, Runnable>) task -> () -> {
                try {
                    callback.apply(task);
                } catch (Exception e) {
                    logger.warn(e, e);
                }
            }).forEach(executor::submit);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

}
