package ru.yandex.chemodan.app.docviewer.adapters.openoffice;

import java.util.Arrays;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import lombok.ToString;
import org.joda.time.Duration;
import oshi.SystemInfo;
import oshi.software.os.OSProcess;
import oshi.software.os.OperatingSystem;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.utils.ProcessUtils2;
import ru.yandex.misc.dataSize.DataSize;

@RequiredArgsConstructor
@ToString
public class OfficeProcessHandle implements OfficeHandle {

    @ToString.Exclude
    private final Process process;

    @ToString.Include
    private String pid() {
        return ProcessUtils2.getProcessPidO(process).map(Object::toString).getOrElse("unknown");
    }

    @Override
    public boolean isAlive() {
        return ProcessUtils2.isAlive(process);
    }

    @Override
    public Option<Integer> getExitCode() {
        return Option.of(process.exitValue());
    }

    @Override
    public Option<DataSize> getMemoryUsage() {
        return ProcessUtils2.getProcessPidO(process).flatMapO(OfficeProcessHandle::getMemoryUsage);
    }

    public static Option<DataSize> getMemoryUsage(int pid) {
        OperatingSystem operatingSystem = new SystemInfo().getOperatingSystem();
        OSProcess[] osProcesses = operatingSystem.getProcesses(0, OperatingSystem.ProcessSort.MEMORY);
        Optional<OSProcess> sofficeOSProcess = Arrays.stream(osProcesses)
                .filter(osProcess -> pid == osProcess.getParentProcessID())
                .findFirst();
        int sofficePid = sofficeOSProcess.map(OSProcess::getProcessID).orElse(pid);
        return Option
                .ofNullable(operatingSystem.getProcess(sofficePid))
                .map(OSProcess::getResidentSetSize)
                .map(DataSize::fromBytes);
    }

    @Override
    public void kill() {
        ProcessUtils2.killProcess(process, Duration.standardSeconds(10));
    }
}
