package ru.yandex.chemodan.app.docviewer.adapters.openoffice;

import java.net.ConnectException;

import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import lombok.RequiredArgsConstructor;
import org.apache.commons.dbcp2.AbandonedTrace;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class OpenOfficeProcess extends AbandonedTrace {

    private static final Logger logger = LoggerFactory.getLogger(OpenOfficeProcess.class);

    private final OfficeHandle handle;
    private final File2 workDirectory;
    private final int port;
    private final DataSize maxResidentSetSize;

    boolean canConnect() {
        logger.debug("Checking if can connect to {}", this);

        boolean connected;
        try {
            OpenOfficeConnection connection = this.newConnection();
            connection.connect();
            connection.disconnect();

            connected = true;
        } catch (ConnectException exc) {
            // expected
            logger.debug("Couldn't connect to office: {}", exc.getMessage());
            connected = false;
        }
        return connected;
    }

    public void close() {
        logger.info("Closing {}", this);
        handle.kill();
    }

    @Override
    protected void finalize() throws Throwable {
        close();
        workDirectory.deleteRecursiveQuietly();
        super.finalize();
    }

    boolean isAliveWithLowRSS() {
        if (!handle.isAlive()) {
            logger.info("{} is not alive", this);
            return false;
        }
        return handle.getMemoryUsage().map(rss -> {
            if (!maxResidentSetSize.gt(rss)) {
                logger.info("{} has reached RSS limit, {} > {}", this, rss.toPrettyString(), maxResidentSetSize.toPrettyString());
                return false;
            } else {
                logger.info("{} is within RSS limit, {} < {}", this, rss.toPrettyString(), maxResidentSetSize.toPrettyString());
                return true;
            }
        }).getOrElse(() -> {
            logger.info("could not determine RSS for {}", this);
            return false;
        });
    }

    SocketOpenOfficeConnection2 newConnection() {
        logger.debug("Opening new connection to {}", this);
        return new SocketOpenOfficeConnection2(port);
    }

    @Override
    public String toString() {
        return String.format("%s, port=%s", handle, port);
    }
}
