package ru.yandex.chemodan.app.docviewer.adapters.openoffice;

import java.io.File;
import java.net.ServerSocket;
import java.util.Arrays;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.PostConstruct;

import lombok.SneakyThrows;
import org.apache.commons.pool2.PooledObject;
import org.apache.commons.pool2.PooledObjectFactory;
import org.apache.commons.pool2.impl.DefaultPooledObject;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.util.HostBasedSwitch;
import ru.yandex.commune.alive2.location.Location;
import ru.yandex.commune.alive2.location.LocationType;
import ru.yandex.inside.porto.PortoClient;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.time.Stopwatch;
import ru.yandex.misc.time.TimeUtils;

public class OpenOfficeProcessFactory implements PooledObjectFactory<OpenOfficeProcess> {

    private static final Logger logger = LoggerFactory.getLogger(OpenOfficeProcessFactory.class);
    @Value("${openoffice.ldpreload.env.variable}")
    private String ldpreloadVariable;
    @Value("${openoffice.soffice.bin}")
    private String command;
    @Value("${openoffice.soffice.start.tries}")
    private int maxStarts;
    @Value("${openoffice.soffice.start.timeout}")
    private Duration startTimeout;
    @Value("${openoffice.soffice.max.resident.set.size}")
    private DataSize maxResidentSetSize;
    @Autowired
    private PortoClient porto;
    @Autowired
    private Location myLocation;

    private final HostBasedSwitch portoEnabled = new HostBasedSwitch("dv-openoffice-porto-enabled-for-hosts");

    static int findFreePort() {
        try (ServerSocket server = new ServerSocket(0)) {
            return server.getLocalPort();
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @Override
    public void activateObject(PooledObject<OpenOfficeProcess> obj) {
        // noop
    }

    @PostConstruct
    public void afterPropertiesSet() {
        checkCommand();
    }

    private void checkCommand() {
        EnvironmentType env = EnvironmentType.getActive();
        if (env == EnvironmentType.DEVELOPMENT || env == EnvironmentType.TESTS) {
            // no openOffice installed on developer's notebooks or in CI
            return;
        }

        final String commandString = command;

        Check.isTrue(StringUtils.isNotEmpty(commandString),
                "Path to 'soffice.bin' file is not specified in bean configuration");

        final File2 commandFile = new File2(commandString);
        Check.isTrue(commandFile.exists(),
                "Path to 'soffice.bin' file  points to non-existing location: ", commandString);
        Check.isTrue(commandFile.getFile().canExecute(),
                "Current user has no rights to execute specified soffice.bin file: ", commandString);
        Check.isTrue(commandFile.isRegular(),
                "Path to 'soffice.bin' file points to non-file location", commandString);
    }

    @Override
    public void destroyObject(PooledObject<OpenOfficeProcess> obj) {
        obj.getObject().close();
    }

    @SneakyThrows
    public OfficeHandle run(String[] commandWithArgument, File workDirectory) {
        if (porto == null || !portoEnabled.get()) {
            ProcessBuilder processBuilder = new ProcessBuilder(commandWithArgument);
            processBuilder.directory(workDirectory);
            processBuilder.environment().put("LD_PRELOAD", ldpreloadVariable);
            return new OfficeProcessHandle(processBuilder.start());
        } else {
            String name = generateContainerName();
            porto.createWeak(name);
            porto.setProperty(name, "command", Stream.of(commandWithArgument).map(s -> String.format("\"%s\"", s)).collect(Collectors.joining(" ")));
            porto.start(name);
            return new OfficeContainerHandle(porto, name);
        }
    }

    private String generateContainerName() {
        String name = "office-" + UUID.randomUUID();
        if (myLocation.getLocationType() == LocationType.YP) {
            // https://st.yandex-team.ru/RTCSUPPORT-16411
            return "self/" + name;
        }

        return name;
    }

    @Override
    public synchronized PooledObject<OpenOfficeProcess> makeObject() {

        /* Preserving directories between restarts for 81-error code return */
        File2 workDirectory = FileUtils.createTempDirectory("soffice", ".tmp");
        File2 userDir = new File2(workDirectory, "user");
        Stopwatch watch = Stopwatch.createAndStart();

        int starts = 0;
        restart:
        while (true) {

            int port = findFreePort();
            final String[] commandWithArgument = new String[]{
                    command,
                    "--headless",
                    "--invisible",
                    "--nodefault",
                    "--nolockcheck",
                    "--nologo",
                    "--nofirststartwizard",
                    "--norestore",
                    "--accept=socket,host=127.0.0.1,port=" + port + ";urp;",
                    "-env:UserInstallation="
                            + UriUtils.toUriString(userDir).replace("file:", "file://") };
            logger.info("Command line and arguments: {}", Arrays.toString(commandWithArgument));

            logger.info("Working directory exists? {}", workDirectory.exists());
            logger.info("User directory exists? {}", userDir.exists());

            starts++;
            OfficeHandle handle = run(commandWithArgument, workDirectory.getFile());

            boolean okay = false;
            try {
                // Disable office output, because it's not very usefull and
                // seems it could create excessive CPU burns
                // https://st.yandex-team.ru/DOCVIEWER-2546
                /*
                ProcessUtils2.stdoutToDebug(process,
                        "Office-" + pid + "-log", "Office [" + pid + "] std output: ", logger);
                ProcessUtils2.errorToWarn(process,
                        "Office-" + pid + "-log", "Office [" + pid + "] error output: ", logger);
                */

                OpenOfficeProcess openOfficeProcess = new OpenOfficeProcess(handle,
                        workDirectory, port, maxResidentSetSize);

                final Instant startDeadline = TimeUtils.now().plus(startTimeout);
                while (startDeadline.isAfterNow()) {
                    if (!handle.isAlive()) {
                        if (handle.getExitCode().isSome(81) && starts < maxStarts) {
                            logger.info(
                                    "Restarting (#{}) in the same working directory after 81 exit code...",
                                    starts);
                            continue restart;
                        }

                        if (starts < maxStarts) {
                            logger.info("Restarting (#{}) in the same working directory "
                                            + "after {} exit code and hope for the best...",
                                    starts, handle.getExitCode());
                            continue restart;
                        }

                        throw new RuntimeException("Unable to start application (unknown reason). "
                                + "Process return code is " + handle.getExitCode() + ".");
                    }

                    if (openOfficeProcess.canConnect()) {
                        okay = true;
                        break;
                    }

                    ThreadUtils.sleep(100);
                }

                if (!openOfficeProcess.canConnect()) {
                    throw new RuntimeException("Application started but unable to connect to it");
                }

                watch.stopAndLog("Started " + openOfficeProcess, logger);

                return new DefaultPooledObject<>(openOfficeProcess);
            } finally {
                if (handle != null && !okay) {
                    handle.kill();
                }
            }
        }
    }

    @Override
    public void passivateObject(PooledObject<OpenOfficeProcess> obj) {
        // noop
    }

    @Override
    public boolean validateObject(PooledObject<OpenOfficeProcess> obj) {
        try {
            return obj.getObject().isAliveWithLowRSS();
        } catch (Exception e) {
            logger.error(e);
            return false;
        }
    }
}
