package ru.yandex.chemodan.app.docviewer.adapters.openoffice;

import java.util.EnumSet;

import javax.annotation.PreDestroy;

import org.apache.commons.pool2.impl.AbandonedConfig;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jmx.export.annotation.ManagedResource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.docviewer.utils.AbstractPoolBasedBean;
import ru.yandex.chemodan.util.ExecUtils2;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

@ManagedResource
public class OpenOfficeProcessPool extends AbstractPoolBasedBean implements MonicaContainer {

    private static final Logger logger = LoggerFactory.getLogger(OpenOfficeProcessPool.class);

    private final GenericObjectPool objectPool;
    private final String poolName;

    public OpenOfficeProcessPool(
            OpenOfficeProcessFactory openOfficeProcessFactory,
            String poolName,
            int removeAbandonedTimeoutInSeconds,
            int maxIdle,
            int maxActive,
            long maxWait,
            long timeBetweenEvictionRunsMillis,
            int numTestsPerEvictionRun,
            long minEvictableIdleTimeMillis)
    {
        AbandonedConfig abandonedConfig = new AbandonedConfig();
        abandonedConfig.setRemoveAbandonedOnBorrow(true);
        abandonedConfig.setRemoveAbandonedOnMaintenance(true);
        abandonedConfig.setRemoveAbandonedTimeout(removeAbandonedTimeoutInSeconds);

        GenericObjectPoolConfig config = new GenericObjectPoolConfig();
        config.setLifo(false);
        config.setMaxIdle(maxIdle);
        config.setMaxWaitMillis(maxWait);
        config.setMinIdle(1);
        config.setMaxTotal(maxActive);
        config.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        config.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        config.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);

        config.setTestOnBorrow(true);
        config.setTestOnReturn(true);
        config.setTestWhileIdle(true);

        stopAllExistingLO();
        GenericObjectPool<OpenOfficeProcess> newObjectPool = new GenericObjectPool<>(openOfficeProcessFactory, config);
        newObjectPool.setAbandonedConfig(abandonedConfig);

        this.objectPool = newObjectPool;
        this.poolName = poolName;
    }

    private void stopAllExistingLO() {
        if (EnumSet.of(EnvironmentType.PRODUCTION, EnvironmentType.TESTING).contains(EnvironmentType.getActive())) {
            try {
                ExecUtils2.executeWithLogging(Cf.list("pkill", "soffice.bin"));
            } catch (Exception e) {
                logger.info("Nothing to stop: {}", e.getMessage());
            }
        }
    }

    private OpenOfficeProcess borrow() {
        logger.debug("Borrowing openOfficeProcess from pool...");

        try {
            return (OpenOfficeProcess) objectPool.borrowObject();
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @PreDestroy
    public void destroy() {
        try {
            objectPool.close();
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @Override
    protected GenericObjectPool getObjectPool() {
        return objectPool;
    }

    private void invalidateQuietly(OpenOfficeProcess process) {
        logger.debug("Quietly invalidating openOfficeProcess... {}", process);

        try {
            objectPool.invalidateObject(process);
        } catch (Exception exc) {
            logger.error("Unable to invalidate process: " + exc, exc);
        }
    }

    public ExtendedOpenOfficeConnection openConnection(int maxRetries) {
        logger.debug("Borrowing connection...");

        final OpenOfficeProcess openOfficeProcess = borrow();

        logger.debug("Got office with {}", openOfficeProcess);

        SocketOpenOfficeConnection2 delegate = null;
        try {
            delegate = openOfficeProcess.newConnection();
            delegate.connect();
        } catch (Exception exc) {
            boolean spurious = delegate != null && delegate.isSpuriouslyDisconnected();
            boolean interrupted = Thread.interrupted();     // clears interrupted flag!
            logger.error("Failed to connect to OOo [spurious=" + spurious + " interrupted=" + interrupted + "]: " + exc, exc);

            invalidateQuietly(openOfficeProcess);

            if (spurious && maxRetries > 0) {
                logger.debug("Fail was spurious, will retry (attempts left: {}).", maxRetries);
                return openConnection(maxRetries - 1);
            } else {
                if (interrupted && !spurious) {
                    // restore flag -- who knows what was the reason
                    Thread.currentThread().interrupt();
                }

                throw ExceptionUtils.translate(exc);
            }
        }

        return new OpenOfficeConnectionWrapper(delegate) {
            @Override
            public void disconnect() {
                logger.debug("OpenOfficeConnectionWrapper::disconnect()");
                try {
                    super.disconnect();
                } finally {
                    returnObjectQuietly(openOfficeProcess);
                }
            }

            @Override
            public void disconnectQuietly(boolean invalidate) {
                logger.debug("OpenOfficeConnectionWrapper::disconnectQuietly()");
                try {
                    super.disconnectQuietly(invalidate);
                } finally {
                    if (invalidate)
                        invalidateQuietly(openOfficeProcess);
                    else
                        returnObjectQuietly(openOfficeProcess);
                }
            }
        };
    }

    private void returnObjectQuietly(OpenOfficeProcess process) {
        logger.debug("Quietly returning openOfficeProcess to pool... {}", process);

        try {
            objectPool.returnObject(process);
        } catch (Exception exc) {
            logger.error("Unable to return object to pool: " + exc, exc);
        }
    }

    @SuppressWarnings("unused")
    @MonicaMetric
    @GroupByDefault
    public int getNumActive() {
        return objectPool.getNumActive();
    }

    @SuppressWarnings("unused")
    @MonicaMetric
    @GroupByDefault
    public int getNumIdle() {
        return objectPool.getNumIdle();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "oo-process-pool",
                new MetricName("oo-process-pool").withSuffix(poolName),
                "Open office process pool"
        );
    }
}
