package ru.yandex.chemodan.app.docviewer.adapters.openoffice;

import java.net.ConnectException;

import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeException;
import com.sun.star.beans.XPropertySet;
import com.sun.star.bridge.XBridge;
import com.sun.star.bridge.XBridgeFactory;
import com.sun.star.comp.helper.Bootstrap;
import com.sun.star.connection.NoConnectException;
import com.sun.star.connection.XConnection;
import com.sun.star.connection.XConnector;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XMultiComponentFactory;
import com.sun.star.ucb.XFileIdentifierConverter;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.chemodan.app.docviewer.utils.ReflectionUtils2;

/**
 * Copy-paste of SocketOpenOfficeConnection that does not hang on spurious disconnects during
 * initialization (see comments below).
 * <p>
 * See DOCVIEWER-310
 * <p>
 * XXX this is a workaround; should find a reason of spurious disconnects
 *
 * @author vavinov
 */
public class SocketOpenOfficeConnection2 implements OpenOfficeConnection {
    public static final String DEFAULT_HOST = "localhost";
    public static final int DEFAULT_PORT = 8100;
    private static final Logger logger = LoggerFactory.getLogger(SocketOpenOfficeConnection2.class);
    private final String connectionString;
    private XComponent bridgeComponent;
    private XMultiComponentFactory serviceManager;
    private XComponentContext componentContext;
    private XBridge bridge;
    private boolean connected = false;
    private boolean expectingDisconnection = false;

    private volatile boolean spuriouslyDisconnected = false;

    public SocketOpenOfficeConnection2() {
        this(DEFAULT_HOST, DEFAULT_PORT);
    }

    public SocketOpenOfficeConnection2(int port) {
        this(DEFAULT_HOST, port);
    }

    public SocketOpenOfficeConnection2(String host, int port) {
        this("socket,host=" + host + ",port=" + port + ",tcpNoDelay=1");
    }

    private SocketOpenOfficeConnection2(String connectionString) {
        this.connectionString = connectionString;
    }

    public synchronized void connect() throws ConnectException {
        logger.debug("connecting");
        try {
            final Thread threadToInterruptOnSpuriousDisconnect = Thread.currentThread();

            XComponentContext localContext = Bootstrap.createInitialComponentContext(null);
            XMultiComponentFactory localServiceManager = localContext.getServiceManager();
            XConnector connector = UnoRuntime.queryInterface(XConnector.class,
                    localServiceManager.createInstanceWithContext("com.sun.star.connection.Connector", localContext));
            XConnection connection = connector.connect(connectionString);
            XBridgeFactory bridgeFactory = UnoRuntime.queryInterface(XBridgeFactory.class,
                    localServiceManager.createInstanceWithContext("com.sun.star.bridge.BridgeFactory", localContext));
            bridge = bridgeFactory.createBridge("", "urp", connection, null);
            bridgeComponent = UnoRuntime.queryInterface(XComponent.class, bridge);
            bridgeComponent.addEventListener(eventObject -> {
                connected = false;
                // XXX possible long output; don't use maxDepth > 2
                String eventObjectStr = ReflectionUtils2.reflectionToStringValueObject(eventObject, 1);
                if (expectingDisconnection) {
                    logger.info("disconnected: {}", eventObjectStr);
                } else {
                    logger.error("disconnected unexpectedly: {}", eventObjectStr);
                    spuriouslyDisconnected = true;

                    // interrupt originating thread to avoid hanging on bridge.getInstance
                    threadToInterruptOnSpuriousDisconnect.interrupt();
                }
                expectingDisconnection = false;
            });

            // spurious disconnect ("disconnected unexpectedly") may occur during bridge#getInstance:
            serviceManager = UnoRuntime.queryInterface(XMultiComponentFactory.class,
                    bridge.getInstance("StarOffice.ServiceManager"));

            XPropertySet properties = UnoRuntime.queryInterface(XPropertySet.class, serviceManager);
            componentContext = UnoRuntime.queryInterface(XComponentContext.class,
                    properties.getPropertyValue("DefaultContext"));
            connected = true;
            logger.info("connected");
        } catch (NoConnectException connectException) {
            throw new ConnectException("connection failed: " + connectionString + ": " + connectException.getMessage());
        } catch (Exception exception) {
            throw new OpenOfficeException("connection failed: " + connectionString, exception);
        }
    }

    public synchronized void disconnect() {
        logger.debug("disconnecting");
        expectingDisconnection = true;
        bridgeComponent.dispose();
    }

    public boolean isSpuriouslyDisconnected() {
        return spuriouslyDisconnected;
    }

    public boolean isConnected() {
        return connected;
    }

    private Object getService(String className) {
        try {
            if (!connected) {
                logger.info("trying to (re)connect");
                connect();
            }
            return serviceManager.createInstanceWithContext(className, componentContext);
        } catch (Exception exception) {
            throw new OpenOfficeException("could not obtain service: " + className, exception);
        }
    }

    public XComponentLoader getDesktop() {
        return UnoRuntime.queryInterface(XComponentLoader.class,
                getService("com.sun.star.frame.Desktop"));
    }

    public XFileIdentifierConverter getFileContentProvider() {
        return UnoRuntime.queryInterface(XFileIdentifierConverter.class,
                getService("com.sun.star.ucb.FileContentProvider"));
    }

    public XBridge getBridge() {
        return bridge;
    }

    public XMultiComponentFactory getRemoteServiceManager() {
        return serviceManager;
    }

    public XComponentContext getComponentContext() {
        return componentContext;
    }

}


