package ru.yandex.chemodan.app.docviewer.adapters.poppler;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.docviewer.convert.result.PageInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.ExecUtils2;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetType;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.image.imageMagick.resize.AndrushaResizeUtils;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author ssytnik
 */
public class PopplerAdapter {
    private static final Logger logger = LoggerFactory.getLogger(PopplerAdapter.class);

    private static final Pattern PAGE_SIZE_PATTERN = Pattern.compile(
            "^Page\\s+(\\d+)\\s+size:\\s+(\\S+) x (\\S+)\\s.*$", Pattern.MULTILINE);
    private static final Pattern PAGE_ROT_PATTERN = Pattern.compile(
            "^Page\\s+(\\d+)\\s+rot:\\s+(\\d+)$", Pattern.MULTILINE);

    private static final float DEFAULT_DPI = 72.f;

    private final DynamicProperty<Boolean> pixelPerfectPages =
            new DynamicProperty<>("poppler-pixel-perfect-pages", false);

    @Value("${poppler.render.dpi}")
    private final int dotsPerInch;

    @Value("${poppler.pdfinfo}")
    private final String pdfinfo;

    @Value("${poppler.pdftoppm}")
    private final String pdftoppm;

    @Value("${poppler.pdfseparate}")
    private final String pdfseparate;

    public PopplerAdapter(String pdfinfoCmd, String pdftoppmCmd, String pdfseparateCmd, int dpi) {
        pdfinfo = pdfinfoCmd;
        pdftoppm = pdftoppmCmd;
        pdfseparate = pdfseparateCmd;
        dotsPerInch = dpi;
    }

    public void convert(final InputStreamSource source, final OutputStreamSource target,
            final int pageIndex, final ResizeOption size,
            final PdfRenderTargetType targetType, final boolean useCropBox)
    {
        Validate.in(targetType, Cf.list(PdfRenderTargetType.JPG, PdfRenderTargetType.PNG));

        FileUtils.withFile(source, sourceFile -> FileUtils.withFile(target, targetFile -> {
            convertImpl(sourceFile, targetFile, pageIndex, size, targetType, useCropBox);
        }));
    }

    public void convertImpl(final File2 sourceFile, final File2 targetFile,
            final int pageIndex, final ResizeOption resize,
            final PdfRenderTargetType targetType, final boolean useCropBox)
    {
        ThreadLocalTimeout.check();
        FileUtils.checkFileForReading(sourceFile);

        runScript(getConvertCommandLine(sourceFile, targetFile, pageIndex, resize, targetType, useCropBox));

        new File2(targetFile.getAbsoluteFile() + "." + targetType.value()).renameTo(targetFile);
    }

    public void separate(final InputStreamSource source, final OutputStreamSource target,
            final int firstPage, final int lastPage)
    {
        FileUtils.withFile(source, sourceFile -> FileUtils.withFile(target, targetFile -> {
            separateImpl(sourceFile, targetFile, firstPage, lastPage);
        }));
    }

    public void separateImpl(final File2 sourceFile, final File2 targetFile,
            final int firstPage, final int lastPage)
    {
        ThreadLocalTimeout.check();
        FileUtils.checkFileForReading(sourceFile);

        runScript(getSeparateCommandLine(sourceFile, targetFile, firstPage, lastPage));
    }

    private ListF<String> getConvertCommandLine(final File2 sourceFile, final File2 targetFileBase,
            final int pageIndex, final ResizeOption resize, final PdfRenderTargetType targetType,
            final boolean useCropBox)
    {
        ListF<String> cmd = Cf.arrayList();

        cmd.add(pdftoppm);

        cmd.addAll(pageRangeArguments(pageIndex, pageIndex));

        cmd.add("-" + (targetType == PdfRenderTargetType.JPG ? "jpeg" : targetType.value()));

        if (useCropBox) {
            cmd.add("-cropbox");
        }

        cmd.add("-r");
        cmd.add(String.valueOf(dotsPerInch));

        if (resize.size.hasWidthOrHeight()) {
            PageInfo pageInfo = getSinglePageInfo(sourceFile, pageIndex);

            if (pixelPerfectPages.get()
                    && resize.size.width.isPresent() && resize.size.height.isPresent()
                    && resize.type == ResizeOption.ResizeType.SCALE)
            {
                int width = resize.size.width.get();
                int height = resize.size.height.get();
                boolean rotated = pageInfo.getRotated().getOrElse(false);

                cmd.add("-scale-to-x");
                cmd.add(String.valueOf(rotated ? height : width));
                cmd.add("-scale-to-y");
                cmd.add(String.valueOf(rotated ? width : height));
            } else {
                int boundingBoxEdge = calcScaleToCoeff(pageInfo, resize);
                cmd.add("-scale-to");
                cmd.add(String.valueOf(boundingBoxEdge));
            }
        }

        cmd.add("-singlefile");

        // source file
        cmd.add(sourceFile.getAbsolutePath()); // e.g. file.pdf

        // destination file pattern, result will differ and is to be renamed afterwards
        cmd.add(targetFileBase.getAbsolutePath()); // e.g. file.pdf.jpg

        return cmd;
    }

    int calcScaleToCoeff(PageInfo pageInfo, ResizeOption resize) {
        Dimension was = new Dimension(ptsToPixels(pageInfo.getWidth().get()), ptsToPixels(pageInfo.getHeight().get()));
        Dimension need = new Dimension(resize.size.width.getOrElse(0), resize.size.height.getOrElse(0));

        Dimension result;
        switch (resize.type) {
            case SCALE:
                result = AndrushaResizeUtils.resizeProportional(was, need);
                break;
            case BOUND:
                result = AndrushaResizeUtils.resizeProportionalMinimizeOnly(was, need);
                break;
            default:
                throw new IllegalArgumentException("Unknown resize option: " + resize.type);
        }
        return Math.max(result.getWidth(), result.getHeight());
    }

    private int ptsToPixels(float pts) {
        return (int) (pts * dotsPerInch / DEFAULT_DPI + 0.5);
    }

    int getSquareBoundingBoxEdge(PageInfo pageInfo, DimensionO size) {
        Dimension boundingBox = pageInfo.getBoundingBox(size);
        return Math.max(boundingBox.getWidth(), boundingBox.getHeight());
    }


    public PageInfo getSinglePageInfo(final File2 sourceFile, final int pageIndex) {
        PageInfo pageInfo = getPagesInfo(sourceFile, pageIndex, pageIndex).getPageInfos().single();
        Check.equals(pageIndex, pageInfo.getIndex().get());
        return pageInfo;
    }

    public PagesInfo getAllPagesInfo(final File2 sourceFile) {
        // TODO validate output pages list length with 'Pages: NNN' line of poppler output
        return getPagesInfo(sourceFile, 1, 9999);
    }

    public PagesInfo getPagesInfo(final File2 sourceFile, final int firstPage, final int lastPage) {
        ThreadLocalTimeout.check();
        FileUtils.checkFileForReading(sourceFile);

        ExecResult result = runScript(getInfoCommandLine(sourceFile, firstPage, lastPage));

        PagesInfo pagesInfo = parsePagesInfo(result.getStdout());

        if (pagesInfo.isEmpty()) {
            throw new RuntimeException("pdfinfo did not return pages info for " + sourceFile);
        }

        return pagesInfo;
    }

    PagesInfo parsePagesInfo(String pdfinfoOutput) {
        MapF<Integer, Boolean> pageRotations = Cf.hashMap();

        final Matcher rotMatcher = PAGE_ROT_PATTERN.matcher(pdfinfoOutput);
        while (rotMatcher.find()) {
            int pageIndex = Integer.parseInt(rotMatcher.group(1));
            int rotAngle = Integer.parseInt(rotMatcher.group(2));
            boolean rotated = rotAngle == 90 || rotAngle == 270;

            pageRotations.put(pageIndex, rotated);
        }

        ListF<PageInfo> pageInfos = Cf.arrayList();

        final Matcher sizeMatcher = PAGE_SIZE_PATTERN.matcher(pdfinfoOutput);
        while (sizeMatcher.find()) {
            int pageIndex = Integer.parseInt(sizeMatcher.group(1));
            float widthPts = Float.parseFloat(sizeMatcher.group(2));
            float heightPts = Float.parseFloat(sizeMatcher.group(3));

            boolean rotated = pageRotations.getOrElse(pageIndex, false);
            if (rotated) {
                float tmp = widthPts;
                widthPts = heightPts;
                heightPts = tmp;
            }

            logger.debug("pdfinfo: page {} is {} x {} pts (was {}rotated)",
                    pageIndex, widthPts, heightPts, (rotated ? "" : "not "));

            pageInfos.add(new PageInfo(pageIndex, widthPts, heightPts, rotated));
        }

        return new PagesInfo(pageInfos);
    }

    private ListF<String> getInfoCommandLine(final File2 sourceFile, final int firstPage, final int lastPage) {
        ListF<String> cmd = Cf.arrayList();

        cmd.add(pdfinfo);

        cmd.addAll(pageRangeArguments(firstPage, lastPage));

        // source file
        cmd.add(sourceFile.getAbsolutePath()); // e.g. file.pdf

        return cmd;
    }

    private ListF<String> getSeparateCommandLine(File2 sourceFile, File2 targetFile, int firstPage, int lastPage) {
        ListF<String> cmd = Cf.arrayList();

        cmd.add(pdfseparate);

        cmd.addAll(pageRangeArguments(firstPage, lastPage));

        cmd.add(sourceFile.getAbsolutePath());
        cmd.add(targetFile.getAbsolutePath());

        return cmd;
    }

    private ListF<String> pageRangeArguments(int first, int last) {
        ListF<String> cmd = Cf.arrayList();

        cmd.add("-f");
        cmd.add(String.valueOf(first));

        cmd.add("-l");
        cmd.add(String.valueOf(last));

        return cmd;
    }

    private ExecResult runScript(ListF<String> cmd) {
        ExecResult result = ExecUtils2.runScript(cmd, false);

        if (result.getCode() != 0) {
            throw new RuntimeException("Poppler didn't complete normally: " + cmd + ", result: " + result.getOutput());
        }

        return result;
    }

}
