package ru.yandex.chemodan.app.docviewer.cleanup;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupDao;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfImageCache;
import ru.yandex.chemodan.app.docviewer.web.framework.PreviewHistoryManager;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author akirakozov
 */
@AllArgsConstructor
public class CleanupManager {

    private static final Logger logger = LoggerFactory.getLogger(CleanupManager.class);

    private final PdfImageCache pdfImageCache;
    private final PdfWarmupDao pdfWarmupDao;
    private final ResultsCleanup resultsCleanup;
    private final UriCleanup uriCleanup;
    private final SessionCleanup sessionCleanup;
    private final StoredUriDao storedUriDao;
    private final UriRightsDao uriRightsDao;
    private final PreviewHistoryManager previewHistoryManager;

    private final ExecutorService executorService = Executors.newCachedThreadPool(new ThreadNameIndexThreadFactory("cleanup-worker"));

    public void cleanupHeavy() {
        cleanupByAgeHeavy(Option.empty());
    }

    public void cleanupLight() {
        cleanupByAgeLight(Option.empty());
    }

    public void cleanup() {
        cleanupByAgeFull(Option.empty());
    }

    public void cleanupHour() {
        cleanupByAgeFull(Option.of(Duration.standardHours(1)));
    }

    public void cleanupByAgeFull(Option<Duration> age) {
        cleanupByAgeLight(age);
        cleanupByAgeHeavy(age);
    }

    public void cleanupByAgeLight(Option<Duration> age) {
        Instant now = TimeUtils.now();
        runCleaning(Cf.list(
                () -> sessionCleanup.cleanup(now, age),
                () -> uriCleanup.cleanup(now, age),
                () -> pdfWarmupDao.cleanup(),
                () -> previewHistoryManager.cleanup(now, age)));
    }

    public void cleanupByAgeHeavy(Option<Duration> age) {
        Instant now = TimeUtils.now();
        runCleaning(Cf.list(
                () -> pdfImageCache.cleanup(now, age),
                () -> resultsCleanup.cleanup(now, age)));
    }

    private void runCleaning(List<Runnable> runnableList) {
        logger.debug("Start cleanup");
        runnableList
                .stream()
                .map(t -> CompletableFuture.runAsync(t, executorService)).collect(Collectors.toList())
                .forEach(f -> {
                    try {
                        f.join();
                    } catch (Exception e) {
                        logger.error("Can't stop", e);
                    }
                });
    }

    public void removeAll() {
        logger.warn("Start removing all");
        Instant now = TimeUtils.now();

        runCleaning(Cf.list(
                () -> sessionCleanup.removeAll(now),
                () -> uriCleanup.removeAll(now),
                () -> resultsCleanup.removeAll(now),
                () -> pdfImageCache.removeAll(now)));


        logger.debug("Finished removing all, took " +
                TimeUtils.millisecondsToSecondsStringToNow(now.getMillis()));
    }

    public int cleanupByUid(PassportUidOrZero uid) {
        ListF<ActualUri> actualUris = uriRightsDao.findUrisAccessedByUid(uid);
        for (ActualUri uri : actualUris) {
            cleanupByActualUri(uri);
        }
        return actualUris.size();
    }

    public boolean cleanupByActualUri(ActualUri uri) {
        Option<StoredUri> storedUriO = storedUriDao.find(uri);
        if (storedUriO.isPresent()) {
            StoredUri storedUri = storedUriO.get();
            if (storedUri.getFileId().isPresent()) {
                String fileId = storedUri.getFileId().get();
                for (StoredUri storedUri2 : storedUriDao.findAllByFileId(fileId)) {
                    uriCleanup.cleanupByActualUri(storedUri2.getUri());
                }
                resultsCleanup.cleanupByFileId(fileId);
                pdfImageCache.cleanupByFileId(fileId);
            } else {
                uriCleanup.cleanupByActualUri(uri);
            }
            return true;
        } else {
            logger.info("Record not found by uri: {}", uri);
            return false;
        }
    }

}
