package ru.yandex.chemodan.app.docviewer.cleanup;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.dao.cleanup.CleanerTaskDao;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author akirakozov
 */
public class CleanupWorker {
    private static final Logger logger = LoggerFactory.getLogger(CleanupWorker.class);

    private static final String VALUE_WORKER_DAILY_LOCK = "worker_daily_lock";
    private static final String VALUE_WORKER_NIGHTLY_LOCK = "worker_nightly_lock";

    @Autowired
    private CleanupManager cleanupManager;

    @Autowired
    private CleanerTaskDao cleanerTaskDao;

    @Value("${cleanup.worker.enabled}")
    private boolean workerEnabled;
    @Value("${cleanup.worker.timeout-daily}")
    private Duration workerTimeoutDaily;
    @Value("${cleanup.worker.timeout-nightly}")
    private Duration workerTimeoutNightly;

    @Scheduled(cron = "${cleanup.worker.daily}")
    public void executeDaily() {
        execute(cleanupManager::cleanupLight, VALUE_WORKER_DAILY_LOCK, workerTimeoutDaily);
    }

    @Scheduled(cron = "${cleanup.worker.nightly}")
    public void executeNigtly() {
        execute(cleanupManager::cleanupHeavy, VALUE_WORKER_NIGHTLY_LOCK, workerTimeoutNightly);
    }

    public void execute(Runnable runnable, String lock, Duration timeout) {

        if (!workerEnabled) {
            logger.warn("Cleanup worker is DISABLED");
            return;
        }

        logger.info("Start clean up of docviewer data cache");

        Option<Instant> lastLockTimeO = getLastLockTime(lock);

        lastLockTimeO.filter(t -> TimeUtils.toDurationToNow(t).isLongerThan(timeout)).ifPresent(t -> {
            logger.debug("Release lock, because it's been working still {}", t);
            releaseLock(lock);
        });

        if (getLock(lock, new Instant())) {
            try {
                runnable.run();
            } finally {
                releaseLock(lock);
            }
        }
    }

    private Option<Instant> getLastLockTime(String lock) {
        return cleanerTaskDao.findTimestampById(lock);
    }

    void releaseLock(String lock) {
        cleanerTaskDao.deleteById(lock);
    }

    boolean getLock(String lock, Instant timestamp) {
        return cleanerTaskDao.saveAttempt(lock, timestamp, HostnameUtils.localHostname());
    }
}
