package ru.yandex.chemodan.app.docviewer.cleanup;

import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import lombok.AllArgsConstructor;
import org.apache.http.client.methods.HttpHead;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.log.DocviewerTskvEvent;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.MulcaHttpClient;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfImageCache;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

@AllArgsConstructor
public class ResultsCleanup {
    private static final Logger logger = LoggerFactory.getLogger(ResultsCleanup.class);

    public final DynamicProperty<Integer> resultsCleanupDaysTtl =
            new DynamicProperty<>("docviewer.results.cleanup.days.ttl", 20);

    //Allows to disable ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfImageCache
    public final DynamicProperty<Boolean> cleanRelatedDataInPlace =
            new DynamicProperty<>("docviewer.results.cleanup.related.inplace", false);

    private final ExecutorService executorService = Executors
            .newCachedThreadPool(new ThreadNameIndexThreadFactory("cleanup-result"));

    private final FileStorage resultHolder;
    private final StoredResultDao storedResultDao;
    private final MulcaClient mulcaClient;
    private final MulcaHttpClient mulcaHttpClient;
    private final PdfImageCache pdfImageCache;
    private final StoredUriDao storedUriDao;
    private final ResultCleanupConfigurationBean cleanupConfigurationBean;

    public void cleanup(Instant now, Option<Duration> ageO) {
        cleanup(now, ageO.getOrElse(Duration.standardDays(resultsCleanupDaysTtl.get())));
    }

    public void removeAll(Instant now) {
        cleanup(now, Duration.ZERO);
    }

    public void cleanup(Instant now, Duration age) {
        Instant maxTimestamp = getMaxTimestampForCleanupFilter(now, age);

        storedResultDao.deleteByLastAccessLessBatch(maxTimestamp, this::cleanupStoredResult);
        // TODO just in case, delete records from storedWordsDao older than maxTimestamp
        // TODO just in case, delete records from fileRightsDao older than maxTimestamp
    }

    public void cleanupByFileId(String fileId) {
        for (TargetType target : TargetType.values()) {
            for (StoredResult storedResult : storedResultDao.find(fileId, target)) {
                cleanupStoredResult(storedResult);
            }
        }
    }

    public void cleanupByFileIdWithLastAccessCheck(String fileId, TargetType targetType, Instant lastAccessLimit) {
        Option<StoredResult> storedResultO = storedResultDao.find(fileId, targetType);
        if (!storedResultO.isPresent()) {
            DocviewerTskvEvent.documentDeleted(fileId, targetType).log();
            return;
        }
        StoredResult storedResult = storedResultO.get();
        if (storedResult.getLastAccess().isAfter(lastAccessLimit)) {
            DocviewerTskvEvent.documentSkipped(fileId, targetType, storedResult.getWeight(), storedResult.getLastAccess()).log();
            return;
        }
        cleanupStoredResult(storedResult);
    }

    public void fixAllResults() {
        storedResultDao.handleEachResult(r -> {
            MulcaId mulcaId = MulcaId.fromMulcaUri(UrlUtils.uri(r.getFileLink().get()));

            try {
                URI uri = mulcaClient.getDownloadUri(mulcaId);
                mulcaHttpClient.execute(new HttpHead(uri));
            } catch (HttpException e) {
                if (e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                    logger.debug("Remove result meta, result not found in storage" + mulcaId);
                    storedResultDao.delete(r.getFileId(), r.getConvertTargetType());
                } else {
                    logger.warn(e.getMessage());
                }
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }

    private Instant getMaxTimestampForCleanupFilter(Instant now, Duration minAge) {
        return now.minus(minAge);
    }

    private void cleanupStoredResult(StoredResult storedResult) {
        if (skipResult(storedResult)) {
            DocviewerTskvEvent.documentSkipped(storedResult.getFileId(), storedResult.getConvertTargetType(),
                    storedResult.getWeight(), storedResult.getLastAccess()).log();
            return;
        }
        cleanupInvalidStoredResult(storedResult);
    }

    public void cleanupInvalidStoredResult(StoredResult storedResult) {
        final String fileId = storedResult.getFileId();
        final TargetType convertTargetType = storedResult.getConvertTargetType();

        List<CompletableFuture> futureList = Cf.arrayList();

        storedResult.getFileLink()
                .ifPresent(s -> futureList.add(CompletableFuture
                .runAsync(() -> resultHolder.delete(resultHolder.toFileLink(s)), executorService)));

        storedResult.getDocumentProperties().getO(DocumentProperties.EXTRACTED_TEXT)
                .ifPresent(s -> futureList.add(CompletableFuture
                .runAsync(() -> resultHolder.delete(resultHolder.toFileLink(s)), executorService)));

        if (cleanRelatedDataInPlace.get()) {
            futureList.add(CompletableFuture.runAsync(() -> pdfImageCache.cleanupByFileId(fileId), executorService));
            futureList.add(CompletableFuture.runAsync(() -> storedUriDao.findAllByFileId(fileId).forEach(s -> storedUriDao.delete(s.getUri()))));
        }
        storedResultDao.delete(fileId, convertTargetType);
        DocviewerTskvEvent.documentDeleted(fileId, convertTargetType).log();

        futureList.forEach(completableFuture -> {
            try {
                completableFuture.get(1, TimeUnit.SECONDS);
            } catch (Exception e) {
                logger.error("can't delete file", e);
            }
        });
    }

    public Instant getCalculatedLastAccessTime(StoredResult storedResult) {
        if (storedResult.getWeight() <= cleanupConfigurationBean.getResultsCleanupWeightThreshold()) {
            return storedResult.getLastAccess();
        }
        return getLastAccessForHeavyItem(storedResult);
    }

    public Option<StoredResult> findPdfFileById(String fileId) {
        return storedResultDao.findAnyOfTypes(fileId, Cf.x(TargetType.values()));
    }

    boolean skipResult(StoredResult storedResult) {
        return storedResult.getWeight() > cleanupConfigurationBean.getResultsCleanupWeightThreshold()
                && getLastAccessForHeavyItem(storedResult).isAfter(Instant.now());
    }

    private Instant getLastAccessForHeavyItem(StoredResult storedResult) {
        return storedResult.getLastAccess()
                .plus(Duration.standardDays(Math.round(cleanupConfigurationBean.getResultsCleanupMultiplier() * storedResult.getWeight())));
    }
}
