package ru.yandex.chemodan.app.docviewer.cleanup;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class UriCleanup {

    private static final Logger logger = LoggerFactory.getLogger(UriCleanup.class);

    private final DynamicProperty<Integer> uriRightsCleanupTtl = new DynamicProperty<>("docviewer.uri.rights.cleanup.ttl.minutes", 120);
    private final DynamicProperty<Integer> storedUriNotErrorsTtl = new DynamicProperty<>("docviewer.stored.uri.cleanup.ttl.minutes", 5 * 1440);

    private final UriRightsDao uriRightsDao;
    private final StoredUriDao storedUriDao;

    public void cleanup(Instant now, Option<Duration> ageO) {
        cleanup(now, ageO.getOrElse(Duration.standardMinutes(uriRightsCleanupTtl.get())),
                ageO.getOrElse(Duration.standardMinutes(storedUriNotErrorsTtl.get())));
    }

    public void removeAll(Instant now) {
        cleanup(now, Duration.ZERO, Duration.ZERO);
    }

    public void cleanup(Instant now, Duration uriRightsAge, Duration storedUriAge) {
        logger.info("cleanup({}, {}, {})", now, uriRightsAge, storedUriAge);
        Instant maxTimestamp = now.minus(uriRightsAge);
        safeRun(() -> uriRightsDao.deleteByTimestampLessBatch(maxTimestamp));
        safeRun(() -> storedUriDao.deleteErrorsByTimestampLessBatch(maxTimestamp));
        safeRun(() -> storedUriDao.deleteByTimestampLessBatch(now.minus(storedUriAge)));
//        storedUriDao.deleteByTimestampLessBatch(maxTimestamp, a -> cleanupByActualUri(a.getUri()));
        // TODO just in case, delete records from uriRightsDao older than maxTimestamp
    }

    public void cleanupByActualUri(final ActualUri uri) {
        safeRun(() -> uriRightsDao.deleteUriRights(uri));
        safeRun(() -> storedUriDao.delete(uri));
    }

    private void safeRun(Runnable r) {
        try {
            r.run();
        } catch (Exception e) {}
    }

}
