package ru.yandex.chemodan.app.docviewer.cleanup.bazinga;

import java.util.function.Consumer;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.config.DocviewerTaskQueueName;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.time.MoscowTime;

public abstract class CleanupProcessorTask<T extends ItemWithLastAccessDay> extends OnetimeTaskSupport<CleanupProcessorTask.Parameters> {

    private YtHelper ytHelper;

    private DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder;

    public CleanupProcessorTask(Parameters parameters) {
        super(parameters);
    }

    public CleanupProcessorTask(YtHelper ytHelper, DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder) {
        super(Parameters.class);
        this.ytHelper = ytHelper;
        this.documentsCleanupPropertiesHolder = documentsCleanupPropertiesHolder;
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) {
        cleanupDocumentsByYt(parameters);
    }

    public void cleanupDocumentsByYt(Parameters parameters) {
        YPath table = YPath.simple(parameters.getTable());
        if (!ytHelper.existsWithRetries(table)) {
            getLogger().debug("Table '{}' does not exist", table);
            return;
        }
        long rowCount = ytHelper.getRowCount(table);
        if (rowCount < parameters.getStartIndex()) {
            getLogger().debug("The row count is less than start index table={} rc={} startIndex={}",
                    table, rowCount, parameters.getStartIndex());
            return;
        }
        Instant lastAccessLimitForDeletion = Instant.now().minus(Duration.standardDays(documentsCleanupPropertiesHolder.getDaysBeforeNowToCheckLastAccessDay()));
        LocalDate lastAccessDayLimit = new LocalDate(lastAccessLimitForDeletion, MoscowTime.TZ);
        ytHelper.runWithRetries(() -> ytHelper.tables().read(
                table.withRange(parameters.getStartIndex(), Math.min(rowCount, parameters.getEndIndex())),
                YTableEntryTypes.bender(getEntityClass()),
                (Consumer<T>) itemWithLastAccessDay ->
                        this.processItem(itemWithLastAccessDay, lastAccessDayLimit, createProcessingContext(lastAccessLimitForDeletion))
        ));
    }

    public void processItem(T itemIdWithLastAccessDay, LocalDate lastAccessDayLimit, ProcessingContext processingContext) {
        try {
            if (!CleanupUtils.isLastAccessDayValid(
                    itemIdWithLastAccessDay.getLastAccessDay(), lastAccessDayLimit)) {
                getLogger().debug("The last access limit has been reached lastAccess='{}' for item={}",
                        itemIdWithLastAccessDay.getLastAccessDay(),
                        itemIdWithLastAccessDay);
            }
        } catch (IllegalArgumentException e) {
            getLogger().debug(e);
            return;
        }
        processItem(itemIdWithLastAccessDay, processingContext);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(10);
    }

    @Override
    public TaskQueueName queueName() {
        return DocviewerTaskQueueName.DOCVIEWER_REGULAR;
    }

    protected ProcessingContext createProcessingContext(Instant lastAccessLimitForDeletion) {
        return new ProcessingContext(lastAccessLimitForDeletion, Cf.hashMap());
    }

    protected abstract Logger getLogger();

    protected abstract Class<T> getEntityClass();

    protected abstract void processItem(T item, ProcessingContext processingContext);

    @Data
    @BenderBindAllFields
    public static class Parameters {

        private final String table;

        private final long startIndex;

        private final long endIndex;

    }

    @Data
    public static class ProcessingContext {

        private final Instant lastAccessLimitForDeletion;

        private final MapF<String, FileIdLastAccessData> fileIdDeletionCache;

        public FileIdLastAccessData addDeletedToCache(String fileId) {
            FileIdLastAccessData fileIdLastAccessData = new FileIdLastAccessData(true, Option.empty());
            fileIdDeletionCache.put(fileId, fileIdLastAccessData);
            return fileIdLastAccessData;
        }

        public FileIdLastAccessData addExistedToCache(String fileId, Instant lastAccessTime) {
            FileIdLastAccessData fileIdLastAccessData = new FileIdLastAccessData(false, Option.of(lastAccessTime));
            fileIdDeletionCache.put(fileId, fileIdLastAccessData);
            return fileIdLastAccessData;
        }
    }

    @Data
    public static class FileIdLastAccessData {

        private final boolean deleted;

        private final Option<Instant> lastAccessTime;
    }
}
