package ru.yandex.chemodan.app.docviewer.cleanup.bazinga;

import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import lombok.Data;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.commons.lang3.mutable.MutableObject;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.docviewer.config.DocviewerTaskQueueName;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.SchedulePeriodic;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.time.MoscowTime;

public abstract class CleanupSchedulerTask<T extends ItemWithLastAccessDay> extends CronTask {

    private final long minutesToRepeat;

    private final YtHelper ytHelper;

    private final BazingaTaskManager bazingaTaskManager;

    private final DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder;

    public CleanupSchedulerTask(long minutesToRepeat, YtHelper ytHelper, BazingaTaskManager bazingaTaskManager,
            DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder)
    {
        this.minutesToRepeat = minutesToRepeat;
        this.ytHelper = ytHelper;
        this.bazingaTaskManager = bazingaTaskManager;
        this.documentsCleanupPropertiesHolder = documentsCleanupPropertiesHolder;
    }

    @Override
    public Schedule cronExpression() {
        return new SchedulePeriodic(minutesToRepeat, TimeUnit.MINUTES);
    }

    @Override
    public void execute(ExecutionContext executionContext) {
        scheduleProcessors();
    }

    @Override
    public TaskQueueName queueName() {
        return DocviewerTaskQueueName.DOCVIEWER_CRON;
    }

    public void scheduleProcessors() {
        YPath dataTable = getTablePathToProcess();
        if (!ytHelper.existsWithRetries(dataTable)) {
            getLogger().debug("The table {} does not exist. Stopping", dataTable);
            return;
        }
        int readyProcessorsCount = bazingaTaskManager
                .getActiveJobs(TaskId.from(getProcessorTaskClass()), SqlLimits.all()).size();
        int maxProcessorsTaskCount = getMaxProcessorTaskCount();
        int tasksCountToSchedule = maxProcessorsTaskCount - readyProcessorsCount;
        if (tasksCountToSchedule <= 0) {
            getLogger().debug("Too many ready processor tasks count={} limit={}", readyProcessorsCount, maxProcessorsTaskCount);
            return;
        }
        YPath offsetAttribute = dataTable.attribute(getOffsetAttributeName());
        long offset = getLongAttributeValue(offsetAttribute);
        long dataTableSize = ytHelper.getRowCount(dataTable);
        if (offset >= dataTableSize) {
            getLogger().debug("The end of the table {} has been reached", dataTable);
            return;
        }
        YPath outputTable = prepareProcessorTables();
        doSchedule(dataTable, offsetAttribute, offset, dataTableSize, tasksCountToSchedule, outputTable);
    }

    public long getLongAttributeValue(YPath attributePath) {
        return !ytHelper.existsWithRetries(attributePath) ? 0L
                : ytHelper.getWithRetries(() -> ytHelper.cypress().get(attributePath).longValue());
    }

    public void setLongAttributeValue(YPath attributePath, long value) {
        ytHelper.runWithRetries(() -> ytHelper.cypress().set(attributePath, value));
    }

    public YPath prepareProcessorTables() {
        LocalDate now = LocalDate.now(MoscowTime.TZ);
        YPath folderToClear = getProcessorTablesPathPrefix().child(now.minusDays(2).toString());
        if (ytHelper.existsWithRetries(folderToClear)) {
            getLogger().debug("Removing old processor table '{}'", folderToClear);
            ytHelper.remove(folderToClear);
        }
        YPath outputTable = getProcessorTablesPathPrefix().child(now.toString())
                .child(String.valueOf(Instant.now().getMillis()));
        if (ytHelper.existsWithRetries(outputTable)) {
            throw new IllegalStateException(String.format("The table '%s' exists", outputTable));
        }
        ytHelper.cypress().create(outputTable, CypressNodeType.TABLE, true);
        return outputTable;
    }

    public void doSchedule(YPath dataTable, YPath offsetAttribute, long offset, long dataTableSize,
            int tasksCountToSchedule, YPath outputTable)
    {
        int perProcessorTaskRowsCount = getPerProcessorTaskRowsCount();
        long limit = Math.min(offset + tasksCountToSchedule * perProcessorTaskRowsCount, dataTableSize);
        YPath pathWithRange = dataTable.withRange(offset, limit);
        YTableEntryType<T> yTableEntryType = YTableEntryTypes.bender(getItemsClass());
        ItemProcessorData itemProcessorData = ItemProcessorData.empty(
                documentsCleanupPropertiesHolder.getDaysBeforeNowToCheckLastAccessDay(), yTableEntryType, outputTable,
                offsetAttribute, perProcessorTaskRowsCount);
        ytHelper.runWithRetries(
                () -> ytHelper.tables().read(
                        pathWithRange,
                        yTableEntryType,
                        (Consumer<T>) itemWithLastAccessDay ->
                                processItemWithLastAccessDay(itemWithLastAccessDay, itemProcessorData)
                ));
        if (!writeProcessorDataToYt(itemProcessorData)) {
            getLogger().debug("Nothing to schedule the table {} has been removed", outputTable);
            return;
        }
        scheduleProcessorTasks(itemProcessorData);
    }

    public void processItemWithLastAccessDay(ItemWithLastAccessDay itemWithLastAccessDay, ItemProcessorData itemProcessorData)
    {
        if (itemProcessorData.isLastAccessLimitReached()) {
            return;
        }
        try {
            if (!CleanupUtils.isLastAccessDayValid(itemWithLastAccessDay.getLastAccessDay(),
                    itemProcessorData.getLimit())) {
                itemProcessorData.lastAccessLimitReached();
                getLogger().debug("The last access limit has been reached lastAccess='{}' item={}",
                        itemWithLastAccessDay.getLastAccessDay(),
                        itemWithLastAccessDay);
                return;
            }
        } catch (IllegalArgumentException e) {
            getLogger().debug(e);
            return;
        }
        itemProcessorData.addItem(itemWithLastAccessDay);
    }

    /**
     * returns true if any document data has been written to YT else false
     * @param itemProcessorData
     * @return
     */
    public boolean writeProcessorDataToYt(ItemProcessorData itemProcessorData) {
        ListF<DocumentIdWithLastAccessDay> documents = itemProcessorData.getItems();
        YPath outputTable = itemProcessorData.getOutputTable();
        if (documents.isEmpty()) {
            ytHelper.remove(outputTable);
            return false;
        }
        ytHelper.runWithRetries(
                () -> ytHelper.tables().write(outputTable, itemProcessorData.getYTableEntryType(), documents)
        );
        return true;
    }

    public void scheduleProcessorTasks(ItemProcessorData itemProcessorData) {
        long outputRowCount = ytHelper.getRowCount(itemProcessorData.getOutputTable());
        MutableObject<Instant> scheduleTime = new MutableObject<>(Instant.now().plus(Duration.standardMinutes(1)));
        for (long i = 0; i < outputRowCount; i += itemProcessorData.getItemsPerProcessorTaskCount()) {
            scheduleTime.setValue(scheduleTime.getValue().plus(Duration.standardSeconds(getProcessorTasksScheduleIntervalInSecs())));
            long startIndex = i;
            RetryUtils.retry(getLogger(), 3, () -> bazingaTaskManager.schedule(
                    createProcessorTaskInstance(new CleanupProcessorTask.Parameters(
                            itemProcessorData.getOutputTable().toString(),
                            startIndex,
                            Math.min(outputRowCount, startIndex + itemProcessorData.getItemsPerProcessorTaskCount())
                    )),
                    scheduleTime.getValue()
            ));
        }
        setLongAttributeValue(itemProcessorData.getOffsetAttribute(),
                getLongAttributeValue(itemProcessorData.getOffsetAttribute()) + outputRowCount);
    }

    protected abstract Logger getLogger();

    protected abstract YPath getTablePathToProcess();

    protected abstract Class<? extends CleanupProcessorTask<T>> getProcessorTaskClass();

    protected abstract int getMaxProcessorTaskCount();

    protected abstract String getOffsetAttributeName();

    protected abstract YPath getProcessorTablesPathPrefix();

    protected abstract int getPerProcessorTaskRowsCount();

    protected abstract Class<T> getItemsClass();

    protected abstract long getProcessorTasksScheduleIntervalInSecs();

    protected abstract CleanupProcessorTask<T> createProcessorTaskInstance(CleanupProcessorTask.Parameters parameters);

    @Data
    public static class ItemProcessorData<T extends ItemWithLastAccessDay> {

        public static <T extends ItemWithLastAccessDay> ItemProcessorData<T> empty(int lastAccessBeforeLimit,
                YTableEntryType<T> yTableEntryType,
                YPath outputTable, YPath offsetAttribute,
                int itemsPerProcessorTaskCount)
        {
            return new ItemProcessorData(Cf.arrayList(), new MutableBoolean(false),
                    LocalDate.now(MoscowTime.TZ).minusDays(lastAccessBeforeLimit), yTableEntryType, outputTable,
                    offsetAttribute, itemsPerProcessorTaskCount);
        }

        private final ListF<T> items;

        private final MutableBoolean lastAccessLimitReached;

        private final LocalDate limit;

        private final YTableEntryType<T> yTableEntryType;

        private final YPath outputTable;

        private final YPath offsetAttribute;

        private final int itemsPerProcessorTaskCount;

        public void lastAccessLimitReached() {
            lastAccessLimitReached.setTrue();
        }

        public boolean isLastAccessLimitReached() {
            return lastAccessLimitReached.booleanValue();
        }

        public void addItem(T item) {
            this.items.add(item);
        }
    }
}
