package ru.yandex.chemodan.app.docviewer.cleanup.bazinga;

import lombok.Data;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultsCleanup;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class DocumentsCleanupProcessorTask extends CleanupProcessorTask<DocumentIdWithLastAccessDay> {

    private static final Logger logger = LoggerFactory.getLogger(DocumentsCleanupProcessorTask.class);

    private ResultsCleanup resultsCleanup;

    public DocumentsCleanupProcessorTask(ResultsCleanup resultsCleanup, YtHelper ytHelper,
                                         DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder)
    {
        super(ytHelper, documentsCleanupPropertiesHolder);
        this.resultsCleanup = resultsCleanup;
    }

    protected DocumentsCleanupProcessorTask(Parameters parameters) {
        super(parameters);
    }

    public Option<DocumentId> parseDocumentId(String documentIdLine) {
        String[] parts = documentIdLine.split(" ");
        if (parts.length != 2) {
            logger.debug("Cannot parse document id '{}'", documentIdLine);
            return Option.empty();
        }
        Option<TargetType> targetTypeO = TargetType.getResolver().valueOfO(parts[1]);
        if (!targetTypeO.isPresent()) {
            logger.debug("Cannot find target type for '{}'", documentIdLine);
            return Option.empty();
        }
        return Option.of(new DocumentId(parts[0], targetTypeO.get()));
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected Class<DocumentIdWithLastAccessDay> getEntityClass() {
        return DocumentIdWithLastAccessDay.class;
    }

    @Override
    protected void processItem(DocumentIdWithLastAccessDay item, ProcessingContext processingContext) {
        parseDocumentId(item.getFileId())
                .ifPresent(documentId -> resultsCleanup.cleanupByFileIdWithLastAccessCheck(
                        documentId.getFileId(),
                        documentId.getTargetType(),
                        processingContext.getLastAccessLimitForDeletion()
                ));
    }

    @Data
    public static class DocumentId {

        private final String fileId;

        private final TargetType targetType;
    }

}
