package ru.yandex.chemodan.app.docviewer.config;

import org.apache.http.client.HttpClient;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.app.docviewer.adapters.batik.BatikAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.freehep.FreeHepAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageAnnotator;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageMagickAdapter2;
import ru.yandex.chemodan.app.docviewer.adapters.openoffice.OpenOfficeProcessFactory;
import ru.yandex.chemodan.app.docviewer.adapters.openoffice.OpenOfficeProcessPool;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.PopplerAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.uniconvertor.UniconvertorAdapter;
import ru.yandex.chemodan.app.docviewer.convert.book.BookCoverCreater;
import ru.yandex.chemodan.app.docviewer.copy.provider.UslugiUrlProvider;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.PracticumApiManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.SchoolbookApiManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.TrackerApiManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.UslugiApiManager;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.MulcaHttpClient;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.commune.image.imageMagick.ImageMagickConfiguration;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaUrlSigner;
import ru.yandex.inside.porto.PortoClient;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
@Configuration
@Import({HttpClientContextConfiguration.class,
        UrlProvidersContextConfiguration.class
})
public class AdaptersContextConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(AdaptersContextConfiguration.class);

    @Bean
    public PortoClient portoClient() {
        try {
            return new PortoClient();
        } catch (RuntimeIoException e) {
            logger.warn("porto not available: " + e);
            return null;
        } catch (Exception e) {
            logger.warn("porto not available", e);
            return null;
        }
    }

    @Bean
    public BatikAdapter batikAdapter() {
        return new BatikAdapter();
    }

    @Bean
    public FreeHepAdapter freeHepAdapter(@Value("${emf.image.max.dim.size}") int maxDimSizeLimit) {
        return new FreeHepAdapter(maxDimSizeLimit);
    }

    @Bean
    public ImageMagickAdapter2 imageMagickAdapter2() {
        return new ImageMagickAdapter2();
    }

    @Bean
    public ImageMagick imageMagick(
            @Value("${imagemagick.convert}") String convertCmd,
            @Value("${imagemagick.max.disk.size}") DataSize maxDiskSize,
            @Value("${imagemagick.max.map.size}")DataSize maxMapSize,
            @Value("${imagemagick.max.memory.size}") DataSize maxMemorySize,
            @Value("${imagemagick.max.area.size}") DataSize maxAreaSize)
    {
        ImageMagickConfiguration config = ImageMagickConfiguration.custom()
                .setReadExifWithinConvert(false)
                .setMaxAreaSize(maxAreaSize)
                .setMaxMapSize(maxMapSize)
                .setMaxMemorySize(maxMemorySize)
                .setMaxDiskSize(maxDiskSize).build();
        return new ImageMagick(convertCmd, config);
    }

    @Bean
    public ImageAnnotator textToImage(@Value("${imagemagick.convert}") String convertCmd) {
        return new ImageAnnotator(convertCmd);
    }

    @Bean
    public BookCoverCreater bookCoverCreater() {
        return new BookCoverCreater();
    }

    @Bean
    public PopplerAdapter popplerAdapter(
            @Value("${poppler.render.dpi}") int dpi,
            @Value("${poppler.pdfinfo}") String pdfinfo,
            @Value("${poppler.pdftoppm}") String pdftoppm,
            @Value("${poppler.pdfseparate}") String pdftoseparate)
    {
        return new PopplerAdapter(pdfinfo, pdftoppm, pdftoseparate, dpi);
    }

    @Bean
    public OpenOfficeProcessPool commonOOProcessPool(
            @Value("${openoffice.common.process.remove.abandoned.timeout.in.seconds}") int removeAbandonedTimeoutInSeconds,
            @Value("${openoffice.common.process.max.idle}") int maxIdle,
            @Value("${openoffice.common.process.max.active}") int maxActive,
            @Value("${openoffice.common.process.max.wait}") Duration maxWait,
            @Value("${openoffice.common.process.time.between.eviction.runs.millis}") long timeBetweenEvictionRunsMillis,
            @Value("${openoffice.common.process.num.tests.per.eviction.run}") int numTestsPerEvictionRun,
            @Value("${openoffice.common.process.min.evictable.idle.time}") Duration minEvictableIdleTime,
            OpenOfficeProcessFactory factory)
    {
        return new OpenOfficeProcessPool(factory, "common", removeAbandonedTimeoutInSeconds, maxIdle, maxActive,
                maxWait.getMillis(), timeBetweenEvictionRunsMillis,
                numTestsPerEvictionRun, minEvictableIdleTime.getMillis());
    }

    @Bean
    public OpenOfficeProcessPool presentationOOProcessPool(
            @Value("${openoffice.presentation.process.remove.abandoned.timeout.in.seconds}") int removeAbandonedTimeoutInSeconds,
            @Value("${openoffice.presentation.process.max.idle}") int maxIdle,
            @Value("${openoffice.presentation.process.max.active}") int maxActive,
            @Value("${openoffice.presentation.process.max.wait}") Duration maxWait,
            @Value("${openoffice.presentation.process.time.between.eviction.runs.millis}") long timeBetweenEvictionRunsMillis,
            @Value("${openoffice.presentation.process.num.tests.per.eviction.run}") int numTestsPerEvictionRun,
            @Value("${openoffice.presentation.process.min.evictable.idle.time}") Duration minEvictableIdleTime,
            OpenOfficeProcessFactory factory)
    {
        return new OpenOfficeProcessPool(factory, "presentation", removeAbandonedTimeoutInSeconds,
                maxIdle, maxActive, maxWait.getMillis(), timeBetweenEvictionRunsMillis,
                numTestsPerEvictionRun, minEvictableIdleTime.getMillis());
    }

    @Bean
    public OpenOfficeProcessFactory openOfficeProcessFactory() {
        return new OpenOfficeProcessFactory();
    }

    @Bean
    public UniconvertorAdapter uniconvertorAdapter() {
        return new UniconvertorAdapter();
    }

    @Bean
    public MulcaClient mulcaClient(
            MulcaHttpClient mulcaHttpClient,
            @Value("${mulca.gate.url}") String gateUrl,
            @Value("${mulca.need.sign.url}") boolean needToSignUrls,
            @Value("${mulca.sign.secret}") String secret,
            @Value("${mulca.url.ttl}") Duration urlTtl,
            @Value("${mulca.service}") String service,
            @Value("${mulca.namespace}") String namespace)
    {
        MulcaClient.Builder builder =
                MulcaClient.custom(mulcaHttpClient, gateUrl).withService(service).withNamespace(namespace);

        if (needToSignUrls) {
            builder = builder.withSigner(new MulcaUrlSigner(secret, urlTtl));
        }

        return builder.build();
    }

    @Bean
    public ArchiveManager archiveManager() {
        return new ArchiveManager();
    }

    @Bean
    public TrackerApiManager trackerApiManager(@Qualifier("trackerApiHttpClient") HttpClient httpClient) {
        return new TrackerApiManager(httpClient);
    }

    @Bean
    public UslugiApiManager uslugiApiManager(@Qualifier("uslugiApiHttpClient") HttpClient httpClient,
            UslugiUrlProvider uslugiUrlProvider) {
        return new UslugiApiManager(httpClient, uslugiUrlProvider);
    }

    @Bean
    public SchoolbookApiManager schoolbookApiManager(@Qualifier("schoolbookApiHttpClient") HttpClient httpClient) {
        return new SchoolbookApiManager(httpClient);
    }

    @Bean
    public PracticumApiManager practicumApiManager(@Qualifier("practicumApiHttpClient") HttpClient httpClient) {
        return new PracticumApiManager(httpClient);
    }
}
