package ru.yandex.chemodan.app.docviewer.config;

import java.util.List;
import java.util.Optional;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowire;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.docviewer.archives.ArchiveContextFactory;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.copy.BlockedUrlChecker;
import ru.yandex.chemodan.app.docviewer.copy.Copier;
import ru.yandex.chemodan.app.docviewer.copy.MimeHelper;
import ru.yandex.chemodan.app.docviewer.copy.SerpUrlChecker;
import ru.yandex.chemodan.app.docviewer.copy.StoredUriManager;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.copy.downloader.CachingFileDownloader;
import ru.yandex.chemodan.app.docviewer.copy.downloader.FileToDownloadViaMulcaClient;
import ru.yandex.chemodan.app.docviewer.copy.downloader.MulcaClientFileDownloader;
import ru.yandex.chemodan.app.docviewer.copy.downloader.SynchronizedFileDownloader;
import ru.yandex.chemodan.app.docviewer.copy.provider.DocumentUrlProvider;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.StorageResourceInfoManager;
import ru.yandex.chemodan.app.docviewer.crypt.TokenManager;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.dao.user.UserDocumentsDao;
import ru.yandex.chemodan.app.docviewer.dao.ydb.YdbContextConfiguration;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeChecker;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeCheckerImpl;
import ru.yandex.chemodan.app.docviewer.states.StartManager;
import ru.yandex.chemodan.app.docviewer.states.StateListenerManager;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.states.StateMachineImpl;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.Digester;
import ru.yandex.chemodan.app.docviewer.utils.MaintenanceFileChecker;
import ru.yandex.chemodan.app.docviewer.utils.TmpDirCleanupWorker;
import ru.yandex.chemodan.app.docviewer.utils.XmlSerializer;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfBoxInitializer;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetTypeHolder;
import ru.yandex.chemodan.app.docviewer.web.backend.StoredResultHelper;
import ru.yandex.chemodan.app.docviewer.web.client.DocviewerForwardClient;
import ru.yandex.chemodan.app.docviewer.web.framework.PreviewHistoryManager;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.mime.LibMagicMimeTypeDetector;
import ru.yandex.chemodan.mime.MimeTypeCorrector;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.misc.version.AppName;

/**
 * @author nshmakov
 */
@Configuration
@Import({
        DocviewerInitContextConfiguration.class,
        AdaptersContextConfiguration.class,
        MongoDaoContextConfiguration.class,
        TempFileCacheContextConfiguration.class,
        HtmlUtilsContextConfiguration.class,
        UrlProvidersContextConfiguration.class,
        LocalSchedulerContextConfiguration.class,
        FontManagerContextConfiguration.class,
        CleanupContextConfiguration.class,
        PdfRenderContextConfiguration.class,
        MulcaFileStorageContextConfiguration.class,
        DvZkContextConfiguration.class,
        HttpClientContextConfiguration.class,

        YdbContextConfiguration.class,
})
public class CoreContextConfiguration {

    @Bean
    public DocviewerForwardClient docviewerForwardClient(
            @Value("${docviewer.forward.client.url}")
            String docviewerUrl,
            @Value("${docviewer.forward.client.auth.key}")
            String authKey)
    {
        return new DocviewerForwardClient(docviewerUrl, authKey, fallbackBackendHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("fallback_backend")
    public HttpClientConfigurator fallbackBackendHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public StateMachine stateMachine() {
        return new StateMachineImpl();
    }

    @Bean
    public StateListenerManager stateListenerManager() {
        return new StateListenerManager();
    }

    @Bean
    public StartManager startManager() {
        return new StartManager();
    }

    @Bean
    public Digester digester() {
        return new Digester();
    }

    @Bean
    public ArchiveContextFactory archiveContextFactory(MaxFileSizeChecker maxFileSizeChecker) {
        return new ArchiveContextFactory(maxFileSizeChecker);
    }

    @Bean
    public MimeDetector mimeDetector() {
        return new MimeDetector();
    }

    @Bean
    public LibMagicMimeTypeDetector libMagicMimeTypeDetector(
            @Value("${file.command.path}") String fileCommandPath,
            @Value("${file.mime.detection.additional-args}") String additionalArgs,
            @Value("${file.mime.detection.timeout}") Duration mimeDetectionTimeout,
            Optional<List<MimeTypeCorrector>> correctors)
    {
        return new LibMagicMimeTypeDetector(
                fileCommandPath,
                Cf.x(additionalArgs.split(",")).map(String::trim),
                mimeDetectionTimeout,
                correctors.map(Cf::x).orElse(Cf.list())
        );
    }

    @Bean
    public MaxFileSizeChecker maxFileSizeChecker() {
        return new MaxFileSizeCheckerImpl();
    }

    @Bean
    public StoredResultHelper storedResultHelper() {
        return new StoredResultHelper();
    }

    @Bean
    public StoredUriManager storedUriManager(StoredUriDao storedUriDao, UriRightsDao uriRightsDao) {
        return new StoredUriManager(storedUriDao, uriRightsDao);
    }

    @Bean
    public Copier copier() {
        return new Copier();
    }

    @Bean
    public StorageResourceInfoManager yaDiskCopierResponseGetter() {
        return new StorageResourceInfoManager();
    }

    @Bean(autowire = Autowire.NO)
    public MulcaClientFileDownloader mulcaClientFileDownloader() {
        return new MulcaClientFileDownloader();
    }

    @Bean(autowire = Autowire.NO)
    public CachingFileDownloader<FileToDownloadViaMulcaClient> cachingFileDownloader(
            MulcaClientFileDownloader mulcaClientFileDownloader)
    {
        return new CachingFileDownloader<>(mulcaClientFileDownloader);
    }

    @Bean
    public SynchronizedFileDownloader<FileToDownloadViaMulcaClient> fileDownloader(
            CachingFileDownloader<FileToDownloadViaMulcaClient> cachingFileDownloader)
    {
        return new SynchronizedFileDownloader<FileToDownloadViaMulcaClient>(cachingFileDownloader);
    }

    @Bean
    public UriHelper uriHelper(List<DocumentUrlProvider> documentUrlProviders, SerpUrlChecker serpUrlChecker,
            @Value("${original.uri.check.disable}") boolean disableOriginalUrlCheck,
            @Value("${internal.client.timeout}") Duration timeout, BlockedUrlChecker urlChecker)
    {
        return new UriHelper(Cf.x(documentUrlProviders), serpUrlChecker, disableOriginalUrlCheck, timeout, urlChecker);
    }

    @Bean
    public BlockedUrlChecker urlChecker(@Value("${docviewer.sba.url}") String sbaUrl,
            @Value("${docviewer.sba.client}") String sbaClient)
    {
        return new BlockedUrlChecker(sbaUrl, sbaClient, sbaHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("sba")
    public HttpClientConfigurator sbaHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public SerpUrlChecker serpUrlChecker(@Value("${serp.key.encoded}") String encodedSerpKey) {
        return new SerpUrlChecker(encodedSerpKey);
    }

    @Bean
    public MimeHelper mimeHelper(@Value("${docviewer.is.old.web.enabled}") boolean isOldWebEnabled) {
        return new MimeHelper(isOldWebEnabled);
    }

    @Bean
    public DiskManager diskManager() {
        return new DiskManager();
    }

    @Bean
    public XmlSerializer xmlSerializer() {
        return new XmlSerializer();
    }

    @Bean
    public TokenManager tokenManager(@Value("${docviewer.token.secret}") String secret, @Value("${docviewer.token.old-secret}") String oldSecret) {
        return new TokenManager(secret, oldSecret);
    }

    @Bean
    public MaintenanceFileChecker maintenanceFileChecker(AppName appName) {
        return new MaintenanceFileChecker(appName);
    }

    @Bean
    public PdfBoxInitializer pdfBoxInitializer() {
        return new PdfBoxInitializer();
    }

    @Bean
    public TmpDirCleanupWorker tmpDirCleanupWorker() {
        return new TmpDirCleanupWorker();
    }

    @Bean
    public PreviewHistoryManager previewHistoryManager(DiskManager diskManager, UserDocumentsDao userDocumentsDao,
            @Qualifier("resultHolder") FileStorage resultHolder, PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder)
    {
        return new PreviewHistoryManager(diskManager, userDocumentsDao, resultHolder, pdfRenderTargetTypeHolder);
    }

}
