package ru.yandex.chemodan.app.docviewer.config;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultCleanupConfigurationBean;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultsCleanup;
import ru.yandex.chemodan.app.docviewer.cleanup.StorageCleanupManager;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.CleanupRoutines;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.DocumentsCleanupDataGenerationCronTask;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.DocumentsCleanupProcessorTask;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.DocumentsCleanupPropertiesHolder;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.DocumentsCleanupSchedulerTask;
import ru.yandex.chemodan.app.docviewer.config.conditions.DocumentsCleanupEnabledCondition;
import ru.yandex.chemodan.app.docviewer.yt.YtPathUtils;
import ru.yandex.chemodan.util.yt.BaseIncrementalLogMrYtRunner;
import ru.yandex.chemodan.util.yt.CustomLogMrYtRunner;
import ru.yandex.chemodan.util.yt.IncrementalLogMrYtRunner;
import ru.yandex.chemodan.util.yt.YqlHelper;
import ru.yandex.chemodan.util.yt.YtCommandParameters;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;

@Import({DocumentsCleanupSettingsConfiguration.class,
        CleanupContextConfiguration.class,
        DocviewerYtDataSourceContextConfiguration.class})
@Configuration
@Conditional(DocumentsCleanupEnabledCondition.class)
public class DocumentsCleanupContextConfiguration {

    @Value("${docviewer.documents-cleanup.yt.operation.retry.count}")
    private int ytOperationRetryCount;

    @Value("${docviewer.documents-cleanup.yt.operation.retry.delay}")
    private Duration ytOperationRetryDelay;

    private final DynamicProperty<ListF<String>> additionalInputs = new DynamicProperty<>("docviewer-documents-cleanup-additional-inputs", Cf.list());

    private final DynamicProperty<Integer> daysToDiscover = new DynamicProperty<>("docviewer-documents-cleanup-days-to-discover", 1);

    private final DynamicProperty<Integer> documentCleanupYtMemoryLimitMb =
            new DynamicProperty<>("docviewer-documents-cleanup-yt-memory-limit[Mb]" ,0);
    private final DynamicProperty<Integer> documentCleanupYtJobCount =
            new DynamicProperty<>("docviewer-documents-cleanup-yt-job-count" ,0);

    @Bean
    public RetryPolicy ytDocumentsCleanupRetryPolicy() {
        return new RetryPolicy()
                .withMaxRetries(ytOperationRetryCount)
                .withDelay(ytOperationRetryDelay.getMillis(), TimeUnit.MILLISECONDS);
    }

    @Bean
    public BaseIncrementalLogMrYtRunner cleanupDocumentsMrYtRunner(@Qualifier("documentsCleanupYtHelper") Yt yt,
                                                                   @Qualifier("ytDocumentsCleanupRetryPolicy") RetryPolicy ytDocumentsCleanupRetryPolicy,
                                                                   ResultCleanupConfigurationBean resultCleanupConfigurationBean)
    {
        return new CustomLogMrYtRunner(yt,
                ytDocumentsCleanupRetryPolicy,
                YtPathUtils.getDocumentsCleanupRootYPath(),
                "collect_documents_by_access_time",
                Cf.list("file_id"),
                CleanupRoutines.class,
                Cf.list(YtPathUtils.getDocviewerTskvLogPath(), YtPathUtils.getDocviewerEventsLogPath()),
                () -> Cf.list(YtPathUtils.getDocumentsCleanupYPath()).plus(additionalInputs.get().map(YPath::simple)),
                YtPathUtils.getDocumentsCleanupTempYPath().withAdditionalAttributes(IncrementalLogMrYtRunner.COMPRESSION_ATTRIBUTES),
                () -> Cf.list(String.valueOf(resultCleanupConfigurationBean.getResultsCleanupWeightThreshold()),
                        String.valueOf(resultCleanupConfigurationBean.getResultsCleanupMultiplier())),
                new YtCommandParameters(documentCleanupYtMemoryLimitMb::get, documentCleanupYtJobCount::get)
        );
    }

    @Bean
    public YtHelper documentsCleanupYtHelper(@Qualifier("docviewerYtClient") Yt ytClient,
            @Qualifier("ytDocumentsCleanupRetryPolicy") RetryPolicy ytDocumentsCleanupRetryPolicy)
    {
        return new YtHelper(ytClient, ytDocumentsCleanupRetryPolicy);
    }

    @Bean
    public CleanupRoutines documentsCleanupRoutines(@Qualifier("cleanupDocumentsMrYtRunner") BaseIncrementalLogMrYtRunner cleanupDocumentsMrYtRunner,
                                                             @Qualifier("documentsCleanupYtHelper") YtHelper documentsCleanupYtHelper, YqlHelper yqlHelper)
    {
        return new CleanupRoutines(cleanupDocumentsMrYtRunner, documentsCleanupYtHelper, yqlHelper,
                Cf.list("file_id", "last_access_day"), Cf.list("last_access_day"), daysToDiscover::get,
                YtPathUtils.getDocumentsCleanupYPath());
    }

    @Bean
    public StorageCleanupManager documentsStorageCleanupManager(@Qualifier("documentsCleanupRoutines") CleanupRoutines documentsCleanupRoutines) {
        return new StorageCleanupManager(documentsCleanupRoutines, YtPathUtils.getDocumentsCleanupTempYPath());
    }

    @Bean
    public DocumentsCleanupDataGenerationCronTask documentsCleanupDataGenerationCronTask(
            @Value("${docviewer.documents-cleanup.cron-schedule}") String cronSchedule,
            @Qualifier("documentsStorageCleanupManager") StorageCleanupManager documentsStorageCleanupManager)
    {
        return new DocumentsCleanupDataGenerationCronTask(cronSchedule, documentsStorageCleanupManager);
    }

    @Bean
    public DocumentsCleanupProcessorTask documentsCleanupProcessorTask(ResultsCleanup resultsCleanup, @Qualifier("documentsCleanupYtHelper") YtHelper ytHelper,
                                                                       DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder)
    {
        return new DocumentsCleanupProcessorTask(resultsCleanup, ytHelper, documentsCleanupPropertiesHolder);
    }

    @Bean
    public DocumentsCleanupSchedulerTask documentsCleanupSchedulerTask(
            @Value("${docviewer.documents-cleanup.schedule-interval}") int scheduleInterval,
            @Qualifier("documentsCleanupYtHelper") YtHelper ytHelper, BazingaTaskManager bazingaTaskManager, DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder)
    {
        return new DocumentsCleanupSchedulerTask(scheduleInterval, ytHelper, bazingaTaskManager, documentsCleanupPropertiesHolder);
    }

}
