package ru.yandex.chemodan.app.docviewer.config;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultsCleanup;
import ru.yandex.chemodan.app.docviewer.cleanup.StorageCleanupManager;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.CleanupRoutines;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.DocumentsCleanupPropertiesHolder;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.ImagesCleanupDataGenerationCronTask;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.ImagesCleanupProcessorTask;
import ru.yandex.chemodan.app.docviewer.cleanup.bazinga.ImagesCleanupSchedulerTask;
import ru.yandex.chemodan.app.docviewer.config.conditions.ImagesCleanupEnabledCondition;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfImageCache;
import ru.yandex.chemodan.app.docviewer.yt.YtPathUtils;
import ru.yandex.chemodan.util.yt.BaseIncrementalLogMrYtRunner;
import ru.yandex.chemodan.util.yt.CustomLogMrYtRunner;
import ru.yandex.chemodan.util.yt.IncrementalLogMrYtRunner;
import ru.yandex.chemodan.util.yt.YqlHelper;
import ru.yandex.chemodan.util.yt.YtCommandParameters;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;

@Import({CleanupContextConfiguration.class,
        DocviewerYtDataSourceContextConfiguration.class,
        DocumentsCleanupSettingsConfiguration.class})
@Configuration
@Conditional(ImagesCleanupEnabledCondition.class)
public class ImagesCleanupContextConfiguration {

    @Value("${docviewer.images-cleanup.yt.operation.retry.count}")
    private int ytOperationRetryCount;

    @Value("${docviewer.images-cleanup.yt.operation.retry.delay}")
    private Duration ytOperationRetryDelay;

    private final DynamicProperty<ListF<String>> additionalInputs = new DynamicProperty<>("docviewer-images-cleanup-additional-inputs", Cf.list());

    private final DynamicProperty<Integer> daysToDiscover = new DynamicProperty<>("docviewer-images-cleanup-days-to-discover", 1);

    private final DynamicProperty<Integer> imagesCleanupYtMemoryLimitMb =
            new DynamicProperty<>("docviewer-images-cleanup-yt-memory-limit[Mb]" ,0);
    private final DynamicProperty<Integer> imagesCleanupYtJobCount =
            new DynamicProperty<>("docviewer-images-cleanup-yt-job-count" ,0);

    @Bean
    public RetryPolicy ytImagesCleanupRetryPolicy() {
        return new RetryPolicy()
                .withMaxRetries(ytOperationRetryCount)
                .withDelay(ytOperationRetryDelay.getMillis(), TimeUnit.MILLISECONDS);
    }

    @Bean
    public BaseIncrementalLogMrYtRunner cleanupImagesMrYtRunner(@Qualifier("imagesCleanupYtHelper") Yt yt,
            @Qualifier("ytImagesCleanupRetryPolicy") RetryPolicy ytImagesCleanupRetryPolicy)
    {
        return new CustomLogMrYtRunner(yt,
                ytImagesCleanupRetryPolicy,
                YtPathUtils.getImagesCleanupRootYPath(),
                "collect_images_by_access_time",
                Cf.list("file_id", "image_sub_id"),
                CleanupRoutines.class,
                Cf.list(YtPathUtils.getDocviewerEventsLogPath()),
                () -> Cf.list(YtPathUtils.getImagesCleanupYPath()).plus(additionalInputs.get().map(YPath::simple)),
                YtPathUtils.getImagesCleanupTempYPath().withAdditionalAttributes(IncrementalLogMrYtRunner.COMPRESSION_ATTRIBUTES),
                Cf::list,
                new YtCommandParameters(imagesCleanupYtMemoryLimitMb::get, imagesCleanupYtJobCount::get)
        );
    }

    @Bean
    public YtHelper imagesCleanupYtHelper(@Qualifier("docviewerYtClient") Yt ytClient,
            @Qualifier("ytImagesCleanupRetryPolicy") RetryPolicy ytImagesCleanupRetryPolicy)
    {
        return new YtHelper(ytClient, ytImagesCleanupRetryPolicy);
    }

    @Bean
    public CleanupRoutines imagesCleanupRoutines(
            @Qualifier("cleanupImagesMrYtRunner") BaseIncrementalLogMrYtRunner cleanupImagesMrYtRunner,
            @Qualifier("imagesCleanupYtHelper") YtHelper imagesCleanupYtHelper,
            YqlHelper yqlHelper)
    {
        return new CleanupRoutines(cleanupImagesMrYtRunner, imagesCleanupYtHelper, yqlHelper,
                Cf.list("file_id", "image_sub_id", "last_access_day"), Cf.list("last_access_day", "file_id"), daysToDiscover::get,
                YtPathUtils.getImagesCleanupYPath());
    }

    @Bean
    public StorageCleanupManager imagesStorageCleanupManager(@Qualifier("imagesCleanupRoutines") CleanupRoutines imagesCleanupRoutines) {
        return new StorageCleanupManager(imagesCleanupRoutines, YtPathUtils.getImagesCleanupTempYPath());
    }

    @Bean
    public ImagesCleanupDataGenerationCronTask imagesCleanupDataGenerationCronTask(
            @Value("${docviewer.images-cleanup.cron-schedule}") String cronSchedule,
            @Qualifier("imagesStorageCleanupManager") StorageCleanupManager imagesStorageCleanupManager)
    {
        return new ImagesCleanupDataGenerationCronTask(cronSchedule, imagesStorageCleanupManager);
    }

    @Bean
    public ImagesCleanupProcessorTask imagesCleanupProcessorTask(ResultsCleanup resultsCleanup,
            @Qualifier("imagesCleanupYtHelper") YtHelper ytHelper,
            DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder,
            PdfImageCache pdfImageCache)
    {
        return new ImagesCleanupProcessorTask(ytHelper, documentsCleanupPropertiesHolder, resultsCleanup, pdfImageCache);
    }

    @Bean
    public ImagesCleanupSchedulerTask imagesCleanupSchedulerTask(
            @Value("${docviewer.images-cleanup.schedule-interval}") int scheduleInterval,
            @Qualifier("imagesCleanupYtHelper") YtHelper ytHelper, BazingaTaskManager bazingaTaskManager,
            DocumentsCleanupPropertiesHolder documentsCleanupPropertiesHolder)
    {
        return new ImagesCleanupSchedulerTask(scheduleInterval, ytHelper, bazingaTaskManager, documentsCleanupPropertiesHolder);
    }
}
