package ru.yandex.chemodan.app.docviewer.convert;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageMagickAdapter2;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageMagickConvertInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.states.PagesInfoHelper;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.ImageFileList;
import ru.yandex.chemodan.app.docviewer.utils.ImageUtils;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author ssytnik
 */
public abstract class AbstractImageConverter extends AbstractConverter implements ImageConverter {
    private static final Logger logger = LoggerFactory.getLogger(AbstractImageConverter.class);

    @Autowired
    private ImageMagickAdapter2 imageMagickAdapter2;
    @Autowired
    private ConvertToHtmlHelper convertToHtmlHelper;

    @Override
    public ConvertResultInfo convertImage(
            InputStreamSource inputImage, String contentType,
            TargetType convertTargetType, OutputStreamSource result,
            Option<String> password)
    {
        if (convertTargetType.isHtmlWithImages() || convertTargetType == TargetType.PREVIEW) {
            Option<Integer> layer = Option.empty();
            if (MimeTypes.MIME_ADOBE_PHOTOSHOP.equals(contentType)) {
                logger.info("File is PSD. Use 0-layer for preview");
                layer = Option.of(0);
            }

            ImageMagickConvertInfo convertInfo = imageMagickAdapter2.convert(inputImage, layer);
            File2 imageFile = convertInfo.getTempFileO().get();
            try {
                ImageFormat format = convertInfo.getFormat();

                if (convertTargetType == TargetType.PREVIEW && format.isWeb() && format != ImageFormat.JPEG) {
                    format = ImageFormat.JPEG;
                    ImageUtils.convertInPlace(imageFile, format);
                }

                File2 renamedImageFile = imageFile.sibling("image-" + Random2.R.nextAlnum(18) +
                        "." + format.getExtension());
                imageFile.renameTo(renamedImageFile);
                imageFile = renamedImageFile;

                Document doc = DocumentHelper.createDocument();
                Element img = doc.addElement("html").addElement("body").addElement("img");
                HtmlPostprocessor.addLocalImageSrc(img, imageFile.getName(), true);

                Option<FileList> images = Option.of(new FileList(imageFile));
                // TODO introduce ConvertResultType.IMAGE and get rid of ConvertToHtmlHelper usage
                convertToHtmlHelper.splitAndPack(doc, result, false, images, false);

                PagesInfo pagesInfo = PagesInfoHelper.toSinglePageInfo(convertInfo.getSize());

                return ConvertResultInfo.builder()
                        .type(ConvertResultType.ZIPPED_HTML)
                        .images(images)
                        .pagesInfo(Option.of(pagesInfo))
                        .build();

            } finally {
                ImageFileList.deleteIfEmpty(imageFile);
            }

        } else {
            throw new IllegalStateException("wrong convert target type: " + convertTargetType);
        }
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType.isHtmlWithImages() || convertTargetType == TargetType.PREVIEW;
    }

}
