package ru.yandex.chemodan.app.docviewer.convert;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.annotation.PostConstruct;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.PopplerAdapter;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.convert.result.ExConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionDao;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey.SessionConvertPassword;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.log.ProcessingDoneInfo;
import ru.yandex.chemodan.app.docviewer.states.ConvertUserException;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.PagesInfoHelper;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.states.UserErrorReultsChecker;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ImageFileList;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.ImageHelper;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfHelper;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.PrioritizedFutureTask;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.app.docviewer.utils.security.ContentSecurityChecker;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.ConcurrentUtils;
import ru.yandex.misc.concurrent.TimeoutRuntimeException;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.ExecutionRuntimeException;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeout.Handle;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.misc.time.TimeUtils;

import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.CONVERT_TIMEOUT;
import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.FILE_IS_EMPTY;
import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.FILE_IS_PASSWORD_PROTECTED;
import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.TOO_MANY_RETRIES;
import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.UNKNOWN_CONVERT_ERROR;
import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.UNSUPPORTED_CONVERTION;
import static ru.yandex.chemodan.app.docviewer.states.ErrorCode.UNSUPPORTED_PASSWORD_PROTECTED;

/**
 * @author vlsergey
 */
public class ConvertManager {

    public static final Logger logger = LoggerFactory.getLogger(ConvertManager.class);

    private Map<String, List<Converter>> converters;

    @Value("${convert.timeout}")
    private Duration timeout;
    @Value("${convert.enabled.targets}")
    private String enabledTargetTypesValue;
    @Autowired
    private Scheduler convertScheduler;

    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;

    // XXX One more circular dependency :(
    @Autowired
    private StateMachine stateMachine;
    @Autowired
    private TemporaryFileCache temporaryFileCache;
    @Autowired
    private PopplerAdapter popplerAdapter;
    @Autowired
    private SessionDao sessionDao;
    @Autowired
    private PdfHelper pdfHelper;
    @Autowired
    private ImageHelper imageHelper;
    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private UserErrorReultsChecker userErrorReulstChecker;

    private SetF<TargetType> enabledTargetTypes;
    @PostConstruct
    private void init() {
        enabledTargetTypes =
                Cf.x(StringUtils.split(enabledTargetTypesValue, ";"))
                        .map(TargetType.getResolver()::valueOf).unique();
    }

    private boolean isPreviewType(String type) {
        return converters.getOrDefault(MimeDetector.normalize(type), Cf.list()).stream().anyMatch(c -> c.isSupported(TargetType.PREVIEW));
    }

    public SetF<String> getPreviewMimeTypes(SetF<String> allMimeTypes) {
        return allMimeTypes.filter(this::isPreviewType);
    }

    public ConvertResultInfo convertCommon(final InputStreamSource source,
            final String detectedContentType, final Converter converter,
            final TargetType convertTargetType, final OutputStreamSource result)
    {
        return convertCommon(source, detectedContentType, converter, convertTargetType, result, Option.empty());
    }

    public ConvertResultInfo convertCommon(final InputStreamSource source,
            final String detectedContentType, final Converter converter,
            final TargetType convertTargetType, final OutputStreamSource result,
            final Option<String> password)
    {
        Validate.in(convertTargetType, enabledTargetTypes);
        Validate.isTrue(source.exists(), "Source '" + source + "' does not exist");

        if (source.lengthO().isSome(0L)) {
            throw new UserException(FILE_IS_EMPTY);
        }

        return FileUtils.withFile(result, resultFile -> {
            try {
                final FileOutputStreamSource outputFileSource = new FileOutputStreamSource(resultFile);

                ExecutorService executor = Executors.newSingleThreadExecutor();
                try {
                    final ConvertResultInfo convertResultInfo;
                    Handle handle = ThreadLocalTimeout.push(timeout);
                    try {
                        final String parentRequestId = RequestIdStack.current().getOrElse("");
                        Future<ConvertResultInfo> future = executor.submit(() -> {
                            RequestIdStack.Handle st = RequestIdStack.pushReplace(parentRequestId);
                            try {
                                return converter.convert(source, detectedContentType, convertTargetType,
                                        outputFileSource, password);
                            } finally {
                                st.popSafely();
                            }
                        });
                        convertResultInfo = ConcurrentUtils.get(future);

                        FileUtils.checkFileForReading(resultFile);
                        if (convertResultInfo.isPdf() && convertTargetType != TargetType.PREVIEW) {
                            return postprocessPdf(resultFile).withPropertiesAdded(
                                    convertResultInfo.getProperties());
                        } else {
                            return convertResultInfo;
                        }
                    } catch (TimeoutRuntimeException e) {
                        throw new ThreadLocalTimeoutException(e);
                    } catch (ExecutionRuntimeException e) {
                        throw e.causeToBeRethrown();
                    } catch (Exception e) {
                        throw ExceptionUtils.translate(e);
                    } finally {
                        handle.popSafely();
                    }
                } finally {
                    executor.shutdownNow();
                }

            } catch (ThreadLocalTimeoutException exc) {
                throw new UserException(CONVERT_TIMEOUT, exc);
            }
        });
    }

    private ConvertResultInfo convert(final InputStreamSource source, final String contentType,
            final List<Converter> typeConverters, final TargetType convertTargetType, final OutputStreamSource result)
    {
        List<RuntimeException> exceptions = new ArrayList<>(0);

        for (Converter converter : typeConverters) {
            if (converter.isSupported(convertTargetType)) {
                try {
                    return convertCommon(source, contentType, converter, convertTargetType, result);
                } catch (RuntimeException exc) {
                    if (exc instanceof UserException) {
                        ErrorCode errorCode = ((UserException) exc).getErrorCode();
                        if (EnumSet.of(FILE_IS_EMPTY, FILE_IS_PASSWORD_PROTECTED, UNSUPPORTED_PASSWORD_PROTECTED).contains(errorCode)) {
                            throw exc;
                        }
                    }
                    logger.debug("Unable to convert {} into {} ({}) using {}: {}",
                            source, result, convertTargetType, converter, exc.toString(), exc);
                    exceptions.add(exc);
                }
            }
        }

        if (exceptions.isEmpty()) {
            throw new UserException(UNSUPPORTED_CONVERTION,
                    "None of converters fit '" + contentType + "' to '" + convertTargetType + "'");
        }

        if (exceptions.size() == 1) {
            throw exceptions.get(0);
        }

        throw new UserException(UNKNOWN_CONVERT_ERROR,
                "None of converters were able to convert " + source + " into " + result + "("
                        + convertTargetType + ")", exceptions);
    }

    ConvertResultInfo convert(InputStreamSource source, String actualContentType,
            final TargetType convertTargetType, OutputStreamSource result)
    {
        final List<Converter> typeConverters = getConvertersFor(actualContentType);
        return convert(source, actualContentType, typeConverters, convertTargetType, result);
    }

    public String getConverterName(Converter converter) {
        String name = converter.getClass().getSimpleName();

        if (converter instanceof PdfToHtmlWrapper) {
            return name + "[" + ((PdfToHtmlWrapper) converter).getDelegateName() + "]";
        } else if (converter instanceof ImageToHtmlWrapper) {
            return name + "[" + ((ImageToHtmlWrapper) converter).getDelegateName() + "]";
        } else {
            return name;
        }
    }

    private ExConvertResultInfo convert(final TargetType convertTargetType,
            final Converter converter, String detectedContentType,
            InputStreamSource source, Option<String> password)
    {
        Validate.isTrue(source.exists(), "Source '" + source + "' does not exist");

        final File2 outputFile = FileUtils.createEmptyTempFile("convertresult", ".tmp");
        try {
            ConvertResultInfo resultInfo = convertCommon(source, detectedContentType,
                    converter, convertTargetType, new FileOutputStreamSource(outputFile),
                    password);
            return ExConvertResultInfo.builder()
                    .resultInfo(resultInfo)
                    .resultFile(outputFile)
                    .converterName(getConverterName(converter)).build();
        } catch (Exception e) {
            outputFile.deleteRecursiveQuietly();
            throw ExceptionUtils.translate(e);
        }
    }

    public ExConvertResultInfo convert(ConvertArgs convertArgs, Option<String> password) {
        Instant startTime = TimeUtils.now();

        try {
            if (userErrorReulstChecker.notAllowed(convertArgs.getFailedAttemptsCount(), convertArgs.getLastTry())) {
                throw new UserException(convertArgs.getErrorCode().getOrElse(TOO_MANY_RETRIES));
            }

            List<Converter> typeConverters = getConvertersFor(convertArgs.getContentType());
            ExConvertResultInfo result = convert(
                    convertArgs.getTargetType(), typeConverters, convertArgs.getContentType(), convertArgs.getLocalCopy().getFile(), password);

            LoggerEventsRecorder.saveConvertEvent(convertArgs.getTargetType(), result.getConverterName(),
                    TimeUtils.toDurationToNow(startTime), convertArgs.getContentType(), convertArgs.getStartInfo().getSource().isWarmUp());
            LoggerEventsRecorder.documentProcessingDone(
                    "onConvertDone", convertArgs.getStartInfo(), new ProcessingDoneInfo(State.AVAILABLE,
                            convertArgs.getTargetType(), convertArgs.getFileId(), result.getConverterName()));
            return result.withDuration(startTime);
        } catch (Exception e) {
            String converterName = e instanceof ConvertUserException ?
                    ((ConvertUserException) e).getConverterName() : "";
            LoggerEventsRecorder.saveConvertFailedEvent(convertArgs.getTargetType(), converterName, e,
                    TimeUtils.toDurationToNow(startTime), convertArgs.getContentType(), convertArgs.getStartInfo().getSource().isWarmUp());
            LoggerEventsRecorder.documentProcessingDone("onConvertDone",
                    convertArgs.getStartInfo(), new ProcessingDoneInfo(State.CONVERTING_ERROR, convertArgs.getTargetType(), convertArgs.getFileId(), converterName));
            throw ExceptionUtils.translate(e);
        } finally {
            convertArgs.getLocalCopy().deleteFileIfPossible();
        }
    }

    private ExConvertResultInfo convert(final TargetType convertTargetType,
            final List<Converter> typeConverters, String contentType, InputStreamSource source, Option<String> password)
    {
        ListF<RuntimeException> exceptions = Cf.arrayList();
        for (Converter converter : typeConverters) {
            if (converter.isSupported(convertTargetType)) {
                try {
                    return convert(convertTargetType, converter, contentType, source, password);
                } catch (RuntimeException exc) {
                    if (exc instanceof UserException) {
                        ErrorCode errorCode = ((UserException) exc).getErrorCode();
                        if (EnumSet.of(FILE_IS_EMPTY, FILE_IS_PASSWORD_PROTECTED, UNSUPPORTED_PASSWORD_PROTECTED).contains(errorCode)) {
                            // no need to check another converters
                            throw ConvertUserException.fromRuntimeException(exc, getConverterName(converter));
                        }
                    }

                    logger.info("Unable to convert into {} using {} because of {}", convertTargetType, converter, exc);
                    exceptions.add(ConvertUserException.fromRuntimeException(exc, getConverterName(converter)));
                }
            } else {
                logger.debug("Skip '{}' because it doesn't support target type '{}'", converter,convertTargetType);
            }
        }

        if (exceptions.isEmpty()) {
            throw new UserException(UNSUPPORTED_CONVERTION,
                    "None of converters fit '" + contentType + "' to '" + convertTargetType + "'");
        } else if (exceptions.size() == 1) {
            throw exceptions.get(0);
        } else {
            throw new UserException(UNKNOWN_CONVERT_ERROR,
                    "Unable to convert into " + convertTargetType, exceptions);
        }
    }

    private void convertAndStoreResult(ConvertArgs convertArgs) {
        stateMachine.onConvertBegin(convertArgs.getFileId(), convertArgs.getTargetType());

        ExConvertResultInfo result = null;
        boolean putToCache = false;

        try {
            ContentSecurityChecker.CHECKER.checkContent(convertArgs.getContentType(), convertArgs.getLocalCopy().getFile());
            Option<String> password = sessionDao.findValidValue(convertArgs.getSessionId(), new SessionConvertPassword(convertArgs.getFileId()));

            result = convert(convertArgs, password);
            FileLink pdfLink = storeResult(convertArgs.getFileId(), result);

            stateMachine.onConvertDone(convertArgs, pdfLink, result, password);

            temporaryFileCache.putInCache(pdfLink, result.getResultFile());

            putToCache = prerenderFirstPagesIfNecessary(convertArgs.getFileId(), convertArgs.getTargetType(), result, pdfLink);

        } catch (Exception e) {
            stateMachine.onConvertError(convertArgs, e);
        } finally {
            if (!putToCache && result != null) {
                result.getResultFile().deleteRecursiveQuietly();
            }
        }
    }

    private boolean prerenderFirstPagesIfNecessary(String fileId, TargetType targetType, ExConvertResultInfo result, FileLink pdfLink) {
        // TODO: prerender HTML_WITH_IMAGES_FOR_MOBILE, if they has suitable sizes
        if (result.getResultInfo().isPdf() && (TargetType.HTML_WITH_IMAGES == targetType
                || TargetType.HTML_WITH_IMAGES_FOR_MOBILE == targetType)) {
            result.getResultInfo().getPagesInfo().ifPresent(info ->
                    pdfHelper.renderFirstBlock(fileId, targetType, info));
            pdfHelper.extractText(fileId, targetType, pdfLink);
            return true;
        }
        return false;
    }

    private FileLink storeResult(String fileId, ExConvertResultInfo result) {
        Instant start = TimeUtils.now();
        Option<FileList> images = result.getResultInfo().getImages();
        try {
            FileLink pdfLink = resultHolder.put(result.getResultFile());
            LoggerEventsRecorder.saveStoreResultEvent(
                    TimeUtils.toDurationToNow(start), pdfLink, result.getResultFile().length());
            images.ifPresent(img -> logger.info("About to upload: {}", img));
            CompletableFuture[] collect = ImageFileList.getFiles(images).stream()
                    .map(file -> imageHelper.addHtmlBackgroundAsync(fileId, file.getName(), file))
                    .toArray(CompletableFuture[]::new);
            try {
                CompletableFuture.allOf(collect).join();
            } catch (Exception e) {
                logger.error("can't upload file");
            }
            return pdfLink;
        } catch (Exception e) {
            LoggerEventsRecorder.saveStoreResultFailedEvent(e, TimeUtils.toDurationToNow(start));
            throw ExceptionUtils.translate(e);
        } finally {
            ImageFileList.delete(images);
        }
    }

    Map<String, List<Converter>> getConverters() {
        return converters;
    }

    private List<Converter> getConvertersFor(String contentType) {
        final List<Converter> typeConverters = getConvertersSafelyFor(contentType);

        if (typeConverters.isEmpty()) {
            String msg = "Unable to find convertor for content type '" + contentType + "'";
            throw new UserException(ErrorCode.UNSUPPORTED_SOURCE_TYPE, new UnsupportedOperationException(msg));
        }

        return typeConverters;
    }

    public ListF<Converter> getConvertersSafelyFor(String contentType) {
        final List<Converter> typeConverters = getConverters().get(contentType);
        return typeConverters != null ? Cf.toList(typeConverters) : Cf.list();
    }

    public boolean isScheduled(final String fileId, final TargetType convertTargetType) {
        return convertScheduler.isScheduled(fileId + ";" + convertTargetType);
    }

    public SetF<TargetType> getEnabledTargetTypes() {
        return enabledTargetTypes;
    }

    private ConvertResultInfo postprocessPdf(final File2 resultFile) {
        return PdfUtils.withExistingDocument(resultFile, true, document -> {
            String title = document.getDocumentInformation().getTitle();

            if (StringUtils.contains(title, resultFile.getAbsolutePath())) {
                title = "";
                document.getDocumentInformation().setTitle(title);
                PdfUtils.write(document, new FileOutputStreamSource(resultFile));
            }

            DocumentProperties documentProperties = DocumentProperties.EMPTY.withProperty(DocumentProperties.TITLE, title);

            PagesInfo pagesInfo = PagesInfoHelper.toPagesInfo(document);

            if (pagesInfo.isEmpty()) {
                pagesInfo = popplerAdapter.getAllPagesInfo(resultFile);
                logger.info("Pdfbox found no pages, fallback to poppler gives info on {} pages", pagesInfo.getCount());
            }

            return ConvertResultInfo.builder().type(ConvertResultType.PDF)
                    .pages(pagesInfo.getCount()).pagesInfo(Option.of(pagesInfo))
                    .properties(documentProperties)
                    .build();
        });
    }

    public PrioritizedFutureTask<String, ConvertResultInfo> scheduleConvert(
            final File2 temporaryFile, final String actualContentType,
            final TargetType convertTargetType, final OutputStreamSource result)
    {
        Validate.in(convertTargetType, enabledTargetTypes);
        FileUtils.checkFileForReading(temporaryFile);

        return convertScheduler.scheduleLocalTask(
                temporaryFile.getAbsolutePath(),
                () -> convert(temporaryFile, actualContentType, convertTargetType, result),
                1);
    }

    public void scheduleConvert(ConvertArgs convertArgs) {
        Validate.in(convertArgs.getTargetType(), enabledTargetTypes);
        convertScheduler.scheduleGlobalTask(
                convertArgs.getFileId() + ";" + convertArgs.getTargetType(),
                () -> {
                    convertAndStoreResult(convertArgs);
                    return null;
                },
                convertArgs.getPriority());
    }

    public void setConverters(Map<String, List<Converter>> converters) {
        this.converters = converters;
    }

    public void setTimeout(Duration timeout) {
        this.timeout = timeout;
    }

    public Duration getTimeout() {
        return timeout;
    }
}
