package ru.yandex.chemodan.app.docviewer.convert;

import java.io.IOException;
import java.net.URI;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.annotation.PostConstruct;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;

import eu.medsea.mimeutil.MimeType;
import eu.medsea.mimeutil.MimeUtil2;
import eu.medsea.mimeutil.detector.ExtensionMimeDetector;
import eu.medsea.mimeutil.detector.MagicMimeMimeDetector;
import org.apache.poi.poifs.filesystem.Entry;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageMagickAdapter2;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.mime.LibMagicMimeTypeDetector;
import ru.yandex.chemodan.util.HostBasedSwitch;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.InputStreamXUtils;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileInputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author vlsergey
 * @author ssytnik
 * @author akirakozov
 */
public class MimeDetector {

    private static final Logger logger = LoggerFactory.getLogger(MimeDetector.class);

    static boolean containsAny(Collection<?> collection, Collection<?> anyOf) {
        for (Object object : anyOf) {
            if (collection.contains(object))
                return true;
        }
        return false;
    }

    public static String normalize(String reported0) {
        if (StringUtils.isEmpty(reported0))
            return MimeTypes.MIME_UNKNOWN;

        String reported = reported0.toLowerCase();

        if (MimeTypes.SYNONYMS_ADOBE_ACROBAT.containsTs(reported))
            return MimeTypes.MIME_PDF;

        if (MimeTypes.SYNONYMS_ADOBE_PHOTOSHOP.containsTs(reported))
            return MimeTypes.MIME_ADOBE_PHOTOSHOP;

        if (MimeTypes.SYNONYMS_CDR.containsTs(reported))
            return MimeTypes.MIME_COREL_DRAW;

        if (MimeTypes.SYNONYMS_CSV.containsTs(reported))
            return MimeTypes.MIME_CSV;

        if (MimeTypes.SYNONYMS_ICO.containsTs(reported))
            return MimeTypes.MIME_ICO;

        if (MimeTypes.SYNONYMS_TEXT.containsTs(reported))
            return MimeTypes.MIME_TEXT_PLAIN;

        if (MimeTypes.SYNONYMS_MICROSOFT_BMP.containsTs(reported))
            return MimeTypes.MIME_MICROSOFT_BMP;

        if (MimeTypes.SYNONYMS_MICROSOFT_EXCEL.containsTs(reported))
            return MimeTypes.MIME_MICROSOFT_EXCEL;

        if (MimeTypes.SYNONYMS_MICROSOFT_POWERPOINT.containsTs(reported))
            return MimeTypes.MIME_MICROSOFT_POWERPOINT;

        if (MimeTypes.SYNONYMS_MICROSOFT_WORD.containsTs(reported))
            return MimeTypes.MIME_MICROSOFT_WORD;

        if (MimeTypes.SYNONYMS_RTF.containsTs(reported))
            return MimeTypes.MIME_RTF;

        if (MimeTypes.SYNONYMS_DJVU.containsTs(reported)) {
            return MimeTypes.MIME_DJVU;
        }

        if (MimeTypes.SYNONYMS_ARCHIVE_ZIP.containsTs(reported))
            return MimeTypes.MIME_ARCHIVE_ZIP;

        return reported;
    }

    public static Function<String, String> normalizeF() {
        return MimeDetector::normalize;
    }

    @Autowired
    private ArchiveManager archiveManager;

    @Autowired
    private ImageMagickAdapter2 imageMagickAdapter;

    @Autowired
    private LibMagicMimeTypeDetector libMagicMimeTypeDetector;

    @Autowired
    private ConvertManager convertManager;

    private MimeUtil2 mimeByContent;

    private MimeUtil2 mimeByExtension;

    private XMLInputFactory xmlInputFactory;

    private final HostBasedSwitch correctMimeTypeWithLibMagic =
            new HostBasedSwitch("docviewer-correct-mimetype-with-libmagic-hosts");

    @PostConstruct
    public void afterPropertiesSet() {
        mimeByContent = new MimeUtil2();
        mimeByExtension = new MimeUtil2();
        xmlInputFactory = XMLInputFactory.newFactory();

        mimeByContent.registerMimeDetector(MagicMimeMimeDetector.class.getName());
        mimeByExtension.registerMimeDetector(ExtensionMimeDetector.class.getName());
    }

    public String getContentTypeFamily(String contentType0) {
        String contentType = normalize(contentType0);

        if (StringUtils.isEmpty(contentType)) {
            return "";
        }
        if (MimeTypes.APPLICATION_X_CORELDRAW_RIFF_COMPRESSED.equals(contentType)
                || MimeTypes.IMAGE_X_IMAGEMAGICK_SUPPORTED.equals(contentType)
                || MimeTypes.MIME_ADOBE_PHOTOSHOP.equals(contentType)
                || MimeTypes.MIME_IMAGE_JPEG.equals(contentType)
                || MimeTypes.MIME_IMAGE_PNG.equals(contentType)
                || MimeTypes.MIME_IMAGE_GIF.equals(contentType)
                || MimeTypes.MIME_IMAGE_SVG.equals(contentType)
                || MimeTypes.MIME_IMAGE_TIFF.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_BMP.equals(contentType)
                || MimeTypes.MIME_IMAGE_X_NIKON_NEF.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_EMF.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_WMF.equals(contentType)
                || MimeTypes.MIME_ICO.equals(contentType))
        {
            return "image";
        }
        if (MimeTypes.MIME_PDF.equals(contentType)) {
            return "pdf";
        }
        if (MimeTypes.MIME_HTML.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_WORD.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_WORDML.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_OOXML_WORD.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_OOXML_WORD_TEMPLATE.equals(contentType)
                || MimeTypes.MIME_OPENDOCUMENT_TEXT.equals(contentType)
                || MimeTypes.MIME_RTF.equals(contentType)
                || MimeTypes.MIME_TEXT_PLAIN.equals(contentType)
                || MimeTypes.MIME_PGP.equals(contentType))
        {
            return "text";
        }
        if (MimeTypes.MIME_FB2.equals(contentType)) {
            return "fb2";
        }
        if (MimeTypes.MIME_EPUB.equals(contentType)) {
            return "epub";
        }
        if (MimeTypes.MIME_DJVU.equals(contentType)) {
            return "djvu";
        }
        if (MimeTypes.MIME_MICROSOFT_POWERPOINT.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_OOXML_POWERPOINT.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_OOXML_POWERPOINT_TEMPLATE.equals(contentType)
                || MimeTypes.MIME_OPENDOCUMENT_PRESENTATION.equals(contentType))
        {
            return "presentation";
        }
        if (MimeTypes.MIME_CSV.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_EXCEL.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_SPREADSHEETML.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_OOXML_EXCEL.equals(contentType)
                || MimeTypes.MIME_MICROSOFT_OOXML_EXCEL_TEMPLATE.equals(contentType)
                || MimeTypes.MIME_OPENDOCUMENT_SPREADSHEET.equals(contentType))
        {
            return "spreadsheet";
        }
        if (MimeTypes.ARCHIVE_X_SEVENZIP_SUPPORTED.equals(contentType)
                || MimeTypes.MIME_ARCHIVE_ZIP.equals(contentType))
        {
            return "archive";
        }
        return "unknown";
    }

    public static boolean isArchive(String mimeType) {
        return MimeTypes.MIME_NORMALIZED_ARCHIVES.containsTs(normalize(mimeType));
    }

    public String getMimeType(InputStreamSource inputStreamSource) {
        String result = getMimeTypeImpl(inputStreamSource);

        logger.debug("MIME type for '{}' will be reported as '{}'", inputStreamSource, result);

        return result;
    }

    public String getMimeType(InputStreamSource inputStreamSource, Set<String> reportedTypes) {
        logger.debug("Reported types: {}", reportedTypes);

        String result = getMimeTypeImpl(inputStreamSource);

        if (MimeTypes.MIME_UNKNOWNS.containsTs(result) || MimeTypes.MIME_ARCHIVE_ZIP.equals(result)) {
            for (String reportedType : reportedTypes) {
                if (!MimeTypes.MIME_UNKNOWNS.containsTs(reportedType)) {
                    String newResult = normalize(reportedType);
                    if (!result.equals(newResult)) {
                        logger.debug("Selected {} instead of {} mime type", reportedType, result);
                        result = newResult;
                        break;
                    }
                    // For low-precedence mime types, we are going to search further.
                    // E.g. microsoft excel type can be replaced by test/csv possibly.
                    if (! MimeTypes.MIME_MICROSOFT_EXCEL.equals(result)) {
                        break;
                    }
                }
            }
        }

        if (StringUtils.equals(MimeTypes.MIME_TEXT_PLAIN, result)
                || StringUtils.equals(MimeTypes.IMAGE_X_IMAGEMAGICK_SUPPORTED, result)) // hack for plain text
        {
            if (containsAny(reportedTypes, MimeTypes.SYNONYMS_CSV)) {
                result = MimeTypes.MIME_CSV;
            } else if (containsAny(reportedTypes, MimeTypes.SYNONYMS_TEXT)) {
                result = MimeTypes.MIME_TEXT_PLAIN;
            }
            logger.debug("Mime type refined to {}", result);
        }
        else if (StringUtils.equals(MimeTypes.MIME_IMAGE_TIFF, result)) {
            if (reportedTypes.contains(MimeTypes.MIME_IMAGE_X_NIKON_NEF)) {
                result = MimeTypes.MIME_IMAGE_X_NIKON_NEF;
            }
            logger.debug("Mime type refined to {}", result);
        } else if (StringUtils.equals(MimeTypes.MIME_XML, result)) {
            if (reportedTypes.contains(MimeTypes.MIME_FB2)) {
                result = MimeTypes.MIME_FB2;
            }
            logger.debug("Mime type refined to {}", result);
        }

        if (convertManager.getConvertersSafelyFor(result).isEmpty()
                && reportedTypes.contains(MimeTypes.MIME_TEXT_PLAIN))
        {
            logger.debug("No converter found for {}, but plain text is reported, so redefining to it", result);
            result = MimeTypes.MIME_TEXT_PLAIN;
        }

        logger.debug("MIME type for '{}' will be reported as '{}'", inputStreamSource, result);

        return result;
    }

    public String getMimeTypeByFilename(String fileName) {
        return MimeTypes.overrideExtensionMimeTypes
                .filterKeys(Cf.String.endsWithF().bind1(fileName.toLowerCase()))
                .values().iterator().nextO()
                .getOrElse(MimeTypes.MIME_UNKNOWN);
    }

    public Function<String, String> getMimeTypeByFilenameF() {
        return this::getMimeTypeByFilename;
    }

    public String getMimeTypeByUri(URI uri) {
        return Option.of(uri.getPath())
                .filter(StringUtils::isNotEmpty)
                .map(this::getMimeTypeByFilename)
                .getOrElse(MimeTypes.MIME_UNKNOWN);
    }

    public Function<URI, String> getMimeTypeByUriF() {
        return this::getMimeTypeByUri;
    }

    private String getMimeTypeImpl(InputStreamSource inputStreamSource) {
        Option<String> mimeTypeO = Option.empty();

        if (inputStreamSource instanceof File2 && correctMimeTypeWithLibMagic.get()) {
            File2 file = (File2) inputStreamSource;
            mimeTypeO = libMagicMimeTypeDetector.detectWithOverrides(file, Option.of(file.getAbsolutePath()))
                    .map(MimeDetector::normalize);
        }

        return mimeTypeO.getOrElse(() -> getMimeTypeByMimeUtil(inputStreamSource));
    }

    private String getMimeTypeByMimeUtil(InputStreamSource inputStreamSource) {
        Validate.notNull(inputStreamSource, "inputStreamSource is null");
        logger.debug("Detecting MIME type for '{}'...", inputStreamSource);

        // XXX ssytnik: determines 'application/msword' for excel file (e.g. one from DOCVIEWER-557)
        final Collection<MimeType> mimeUtilResult = getMimeTypes(inputStreamSource);
        String result = mimeUtilResult.stream().findFirst().map(MimeType::toString).orElse(MimeTypes.MIME_UNKNOWN);

        result = normalize(result);

        if (MimeTypes.MIME_XML.equals(result)) {
            result = normalize(getXmlType(inputStreamSource));
        }

        if (MimeTypes.MIME_XML.equals(result) || MimeTypes.MIME_UNKNOWN.equals(result)) {
            if (imageMagickAdapter.isSupported(inputStreamSource)) {
                result = MimeTypes.IMAGE_X_IMAGEMAGICK_SUPPORTED;
            }
        }

        if (MimeTypes.MIME_MICROSOFT_WORD.equals(result)) {
            result = normalize(getOle2Type(inputStreamSource));
        }
        if (MimeTypes.MIME_ARCHIVE_ZIP.equals(result)) {
            result = normalize(getZipMimeType(inputStreamSource));
        }
        if (MimeTypes.MIME_UNKNOWN.equals(result)) {
            if (archiveManager.isArchive(inputStreamSource)) {
                result = MimeTypes.ARCHIVE_X_SEVENZIP_SUPPORTED;
            }
        }

        return result;
    }

    @SuppressWarnings("unchecked")
    private Collection<MimeType> getMimeTypes(InputStreamSource inputStreamSource) {
        final Collection<MimeType> mimeUtilResult;
        if (inputStreamSource instanceof FileInputStreamSource) {
            mimeUtilResult = mimeByContent.getMimeTypes(((FileInputStreamSource) inputStreamSource).getFile());
        } else if (inputStreamSource instanceof UrlInputStreamSource) {
            mimeUtilResult = mimeByContent.getMimeTypes(((UrlInputStreamSource) inputStreamSource).getUrl());
        } else {
            mimeUtilResult = mimeByContent.getMimeTypes(inputStreamSource.readBytes());
        }
        return mimeUtilResult;
    }

    private String getXmlType(InputStreamSource inputStreamSource) {
        return inputStreamSource.readX(getXmlTypeF());
    }

    private Function<InputStreamX, String> getXmlTypeF() {
        return (IoFunction<InputStreamX, String>) a -> {
            try {
                XMLStreamReader reader = xmlInputFactory.createXMLStreamReader(a);
                int nextEvent = reader.nextTag();
                if (nextEvent != XMLStreamReader.START_ELEMENT)
                    return MimeTypes.MIME_XML;

                final QName tagName = reader.getName();

                if (StringUtils.equals("wordDocument", tagName.getLocalPart())
                        && StringUtils.equals(
                                "http://schemas.microsoft.com/office/word/2003/wordml",
                                tagName.getNamespaceURI()))
                {
                    return MimeTypes.MIME_MICROSOFT_WORDML;
                }
                if (StringUtils.equals("Workbook", tagName.getLocalPart())
                        && StringUtils.equals("urn:schemas-microsoft-com:office:spreadsheet",
                                tagName.getNamespaceURI()))
                {
                    return MimeTypes.MIME_MICROSOFT_SPREADSHEETML;
                }

                if (StringUtils.equals("FictionBook", tagName.getLocalPart())
                        && StringUtils.equals("http://www.gribuser.ru/xml/fictionbook/2.0",
                                tagName.getNamespaceURI()))
                {
                    return MimeTypes.MIME_FB2;
                }

                if (StringUtils.equals("svg", tagName.getLocalPart())
                        && StringUtils.equals("http://www.w3.org/2000/svg", tagName.getNamespaceURI()))
                {
                    return MimeTypes.MIME_IMAGE_SVG;
                }
            } catch (Exception exc) {
                logger.warn("Unable to check if XML is Office XML type: " + exc, exc);
            }
            return MimeTypes.MIME_XML;
        };
    }

    private String getOle2Type(InputStreamSource inputStreamSource) {
        try {
            return inputStreamSource.readX(inputStream -> {
                try {
                    POIFSFileSystem poifsFileSystem = new POIFSFileSystem(inputStream);

                    Set<String> entryNames = new HashSet<>();
                    Iterator<Entry> iterator = poifsFileSystem.getRoot().getEntries();
                    while (iterator.hasNext()) {
                        Entry entry = iterator.next();
                        entryNames.add(entry.getName());
                    }

                    if (entryNames.contains("WordDocument")) {
                        return MimeTypes.MIME_MICROSOFT_WORD;
                    }

                    if (entryNames.contains("PowerPoint Document")) {
                        return MimeTypes.MIME_MICROSOFT_POWERPOINT;
                    }

                    if (entryNames.contains("Workbook") || entryNames.contains("WORKBOOK") || entryNames.contains("Book")) {
                        return MimeTypes.MIME_MICROSOFT_EXCEL;
                    }

                    return MimeTypes.MIME_UNKNOWN;
                } catch (Exception exc) {
                    throw ExceptionUtils.translate(exc);
                }
            });
        } catch (Exception exc) {
            logger.warn("Unable to detect OLE type of '" + inputStreamSource + "': " + exc, exc);
            return MimeTypes.MIME_UNKNOWN;
        }
    }

    private String getZipMimeType(final InputStreamSource inputStreamSource) {
        try {

            return FileUtils.withZipFile(inputStreamSource, zipFile -> {
                try {
                    Option<String> result = getMimeTypeFromZipDirectly(zipFile);
                    if (result.isPresent()) {
                        return result.get();
                    }

                    result = getMimeTypeFromCorelDrawZip(zipFile);
                    if (result.isPresent()) {
                        return result.get();
                    }

                    result = OpenXmlZipMimeTypeDetector.getMimeType(zipFile);
                    if (result.isPresent()) {
                        return result.get();
                    }
                } catch (IOException exc) {
                    logger.warn("Unable to detect mime type of '" + inputStreamSource + "': " + exc, exc);
                }

                return MimeTypes.MIME_ARCHIVE_ZIP;
            });

        } catch (UserException e) {

            // Work around java.util.zip behavior: https://jira.yandex-team.ru/browse/DOCVIEWER-1027
            if (e.getErrorCode() == ErrorCode.FILE_IS_PASSWORD_PROTECTED) {
                return MimeTypes.MIME_ARCHIVE_ZIP;
            } else {
                throw e;
            }

        }
    }

    private static Option<String> getMimeTypeFromZipDirectly(ZipFile zipFile) throws IOException {
        ZipEntry zipEntry = zipFile.getEntry("mimetype");
        if (zipEntry != null) {
            String content = InputStreamXUtils.wrap(
                    zipFile.getInputStream(zipEntry)).readString();

            content = content.replaceAll("\\r|\\n", "");
            return StringUtils.notEmptyO(content);
        } else {
            return Option.empty();
        }
    }

    private static Option<String> getMimeTypeFromCorelDrawZip(ZipFile zipFile) throws IOException {
        ZipEntry zipEntry = zipFile.getEntry("metadata/metadata.xml");
        if (zipEntry != null) {
            String content = InputStreamXUtils.wrap(
                    zipFile.getInputStream(zipEntry)).readString();

            if (!StringUtils.isEmpty(content)
                    && content.contains("<ProductName>CorelDRAW X")) // at least, X4 and X5
            {
                return Option.of(MimeTypes.APPLICATION_X_CORELDRAW_RIFF_COMPRESSED);
            }
        }
        return Option.empty();
    }

}
