package ru.yandex.chemodan.app.docviewer.convert;

import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.dom4j.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.misc.io.InputStreamXUtils;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * http://msdn.microsoft.com/en-us/library/bb266220(v=office.12).aspx
 * @author akirakozov
 */
public class OpenXmlZipMimeTypeDetector {
    private static final Logger logger = LoggerFactory.getLogger(OpenXmlZipMimeTypeDetector.class);

    private static final String DOC_CONTENT_TYPE =
            "application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml";
    private static final String PPT_CONTENT_TYPE =
            "application/vnd.openxmlformats-officedocument.presentationml.presentation.main+xml";
    private static final String DOC_TEMPLATE_CONTENT_TYPE =
            "application/vnd.openxmlformats-officedocument.wordprocessingml.template.main+xml";

    private static final String WORKSHEET_CONTENT_TYPE =
            "application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml";
    private static final String BINARY_EXCEL_CONTENT_TYPE =
            "application/vnd.ms-excel.sheet.binary.macroEnabled.main";
    private static final String SLIDE_CONTENT_TYPE =
            "application/vnd.openxmlformats-officedocument.presentationml.slide+xml";

    private static String baseXPathExpr(String mainTag, String attrName, String attrValue) {
        return String.format("/*[name()='Types']/*[name()='%s' and @%s='%s']", mainTag, attrName, attrValue);
    }

    private static boolean checkContentType(Document doc, String baxeXPath, String contentType) {
        return contentType.equals(doc.valueOf(baxeXPath + "/@ContentType"));
    }

    private static boolean existsByXPathExpr(Document doc, String baxeXPath) {
        return Boolean.valueOf(doc.valueOf("boolean(" + baxeXPath + ")"));
    }

    public static Option<String> getMimeType(ZipFile zipFile) {
        ZipEntry zipEntry = zipFile.getEntry("[Content_Types].xml");
        if (zipEntry != null) {
            try {
                Document doc = Dom4jUtils.read(InputStreamXUtils.wrap(zipFile.getInputStream(zipEntry)).readBytes());


                if (checkContentType(doc, baseXPathExpr("Override", "PartName", "/word/document.xml"), DOC_CONTENT_TYPE)
                   || checkContentType(doc, baseXPathExpr("Default", "Extension", "xml"), DOC_CONTENT_TYPE))
                {
                    return Option.of(MimeTypes.MIME_MICROSOFT_OOXML_WORD);
                }

                if (checkContentType(doc, baseXPathExpr("Default", "Extension", "bin"), BINARY_EXCEL_CONTENT_TYPE)) {
                    return Option.of(MimeTypes.overrideExtensionMimeTypes.getTs(".xlsb"));
                }

                if (existsByXPathExpr(doc, baseXPathExpr("Override", "ContentType", WORKSHEET_CONTENT_TYPE)))
                {
                    return Option.of(MimeTypes.MIME_MICROSOFT_OOXML_EXCEL);
                }

                if (existsByXPathExpr(doc, baseXPathExpr("Override", "ContentType", SLIDE_CONTENT_TYPE))
                    || checkContentType(doc, baseXPathExpr("Override", "PartName", "/ppt/presentation.xml"), PPT_CONTENT_TYPE))
                {
                    return Option.of(MimeTypes.MIME_MICROSOFT_OOXML_POWERPOINT);
                }

                if (checkContentType(doc, baseXPathExpr("Override", "PartName", "/word/document.xml"), DOC_TEMPLATE_CONTENT_TYPE))
                {
                    return Option.of(MimeTypes.MIME_MICROSOFT_OOXML_WORD_TEMPLATE);
                }

            } catch (Exception exc) {
                logger.debug("Exception on trying to checking if document is OOXML: ", exc);
            }
        }

        return Option.empty();
    }

}
