package ru.yandex.chemodan.app.docviewer.convert;

import org.apache.pdfbox.pdmodel.PDDocument;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.PdfEncodingFixer;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.FileOutputStreamSource;

/**
 * @author vlsergey
 */
public class PdfToHtmlWrapper extends AbstractConverter {

    private static final Logger logger = LoggerFactory.getLogger(PdfToHtmlWrapper.class);

    private final Converter delegate;

    private boolean handleTextConvertion;

    @Autowired
    private PdfEncodingFixer pdfEncodingFixer;


    public PdfToHtmlWrapper(Converter delegate) {
        super();
        this.delegate = delegate;
        validateDelegate();
    }

    private void validateDelegate() {
        if (delegate == null) {
            throw new IllegalStateException("Delegate is not set");
        }
        if (!delegate.isSupported(TargetType.PDF)) {
            throw new IllegalStateException("Delegate '" + delegate
                    + "' MUST support converting to PDF");
        }
        if (!delegate.isSupported(TargetType.PLAIN_TEXT)) {
            handleTextConvertion = true;
        }
        if (delegate.isSupported(TargetType.HTML_ONLY)) {
            logger.debug("Delegate '{}' SHOULD NOT support converting to HTML_ONLY", delegate);
        }
        if (delegate.isSupported(TargetType.HTML_WITH_IMAGES) || delegate.isSupported(TargetType.HTML_WITH_IMAGES_FOR_MOBILE))
        {
            logger.debug("Delegate '{}' SHOULD NOT support converting to HTML_WITH_IMAGES*",
                    delegate);
        }
    }

    @Override
    public ConvertResultInfo doConvert(final InputStreamSource input, final String contentType,
            TargetType targetType, final OutputStreamSource result, final Option<String> password)
    {
        switch (targetType) {
            case HTML_ONLY:
            case HTML_WITH_IMAGES:
            case HTML_WITH_IMAGES_FOR_MOBILE:
                return doConvertToHtml(input, contentType, result, password);
            case PLAIN_TEXT:
                if (handleTextConvertion) {
                    doConvertToPlainText(input, contentType, result, password);
                    return ConvertResultInfo.builder().type(ConvertResultType.PLAIN_TEXT).build();
                }
                //$FALL-THROUGH$
            default:
                return delegate.convert(input, contentType, targetType, result, password);
        }
    }

    private ConvertResultInfo doConvertToHtml(final InputStreamSource input,
            final String contentType, final OutputStreamSource result,
            final Option<String> password)
    {
        return delegate.convert(input, contentType, TargetType.PDF, result, password);
    }

    private void doConvertToPlainText(final InputStreamSource input, final String contentType,
            final OutputStreamSource result, final Option<String> password)
    {
        FileUtils.withEmptyTemporaryFile("result", ".pdf", tempResultFile -> {
            delegate.convert(input, contentType, TargetType.PDF,
                    new FileOutputStreamSource(tempResultFile), password);
            result.write(PdfUtils.withExistingDocument(tempResultFile, true, pdfEncodingFixer
                    .fixFontsHandler().asFunctionReturnParam().andThen(
                            (Function<PDDocument,String>) PdfUtils::stripText)));
        });
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        if (convertTargetType.isHtml()) {
            return true;
        }
        if (convertTargetType == TargetType.PLAIN_TEXT && handleTextConvertion) {
            return true;
        }
        return delegate.isSupported(convertTargetType);
    }

    public String getDelegateName() {
        return delegate.getClass().getSimpleName();
    }

    @Override
    public String toString() {
        return getClass().getName() + "[" + delegate + "]";
    }
}
