package ru.yandex.chemodan.app.docviewer.convert.archive;

import org.dom4j.Branch;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.archives.ArchiveContextFactory;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.commune.archive.ArchiveContext;
import ru.yandex.commune.archive.ArchiveEntry;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.archive.ArchiveListing.NameAndEntry;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.commune.archive.TreeNode;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author ssytnik
 */
public class ArchiveConverter extends AbstractConverter {

    @Autowired
    private ArchiveManager archiveManager;

    @Autowired
    private ArchiveContextFactory archiveContextFactory;

    @Autowired
    private MimeDetector mimeDetector;

    // XXX FIXME: Circular dependency!
    @Autowired
    private ConvertManager convertManager;

    @Override
    public ConvertResultInfo doConvert(
            InputStreamSource input, String contentType,
            TargetType convertTargetType, OutputStreamSource result,
            Option<String> password)
    {
        ArchiveContext context = archiveContextFactory.archiveListingContext(password);
        ArchiveListing listing = archiveManager.listArchive(input, context);
        TreeNode<NameAndEntry> listingTree = listing.toEntriesTree();

        Document document = listingTreeToDocument(listingTree);
        Dom4jUtils.write(document, result);

        return ConvertResultInfo.builder().type(ConvertResultType.ARCHIVE_LISTING).build();
    }

    @Override
    public boolean isSupported(TargetType targetType) {
        return targetType.isHtmlWithImages();
    }

    private Document listingTreeToDocument(TreeNode<NameAndEntry> tree) {
        Document doc = DocumentHelper.createDocument();
        addSubtree(doc, tree, true);
        return doc;
    }

    private void addSubtree(Branch branch, TreeNode<NameAndEntry> node, boolean isRoot) {
        Option<NameAndEntry> valueO = node.getValueO();
        Option<ArchiveEntry> entryO = valueO.filterMap(NameAndEntry::getEntry);

        boolean isFolder = !isRoot && (entryO.isEmpty() || entryO.get().isFolder());
        boolean hasFile = entryO.isPresent() && !entryO.get().isFolder();

        Element el = branch.addElement(isRoot ? "archive" : isFolder ? "folder" : "file");
        Option<String> nameO = valueO.map(NameAndEntry::getName);
        nameO.ifPresent(name -> el.addAttribute("name", name));
        if (hasFile) {
            // TODO root node for single-file .gz and .bz2 archives
            // contains empty path, thus file is treated non-viewable
            String archiveFilePath = entryO.get().getPath();
            String mimeType = mimeDetector.getMimeTypeByFilename(nameO.getOrElse(archiveFilePath));
            boolean isViewable = !convertManager.getConvertersSafelyFor(mimeType).isEmpty();

            el.addAttribute("path", archiveFilePath);
            entryO.get().getSize().ifPresent(size -> el.addAttribute("size", String.valueOf(size)));
            el.addAttribute("mime-type", mimeType);
            addAttribute(el, "viewable", isViewable);
            addAttribute(el, "encrypted", entryO.get().isPassworded());
            addAttribute(el, "archive", MimeDetector.isArchive(mimeType));
        }

        for (TreeNode<NameAndEntry> child : node.getChildren().sortedBy(foldersFirstF())) {
            addSubtree(el, child, false);
        }
    }

    // should not be applied to the root node which may not have entry value
    private static Function<TreeNode<NameAndEntry>, String> foldersFirstF() {
        return a -> {
            Option<ArchiveEntry> entryO = a.getValue().getEntry();
            return (entryO.isEmpty() || entryO.get().isFolder() ? "0" : "1") + "-" + a.getValue().getName();
        };
    }

    private static void addAttribute(Element el, String name, boolean value) {
        if (value) {
            el.addAttribute(name, "1");
        }
    }
}
