package ru.yandex.chemodan.app.docviewer.convert.book;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.AnnotateImageOptions;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageAnnotator;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.TextSplitOptions;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.TextSplitter;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.misc.io.cl.ClassLoaderUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.random.Random2;

/**
 * @author akirakozov
 */
public class BookCoverCreater {

    private static final int COVERS_COUNT = 12;

    // hard-coded designer decision CHEMODAN-11951
    private static final SetF<Integer> COVERS_WITH_BLACK_FONT = Cf.hashSet(2, 5, 7, 8, 9);

    @Value("${book.cover.font}")
    private String font;

    @Autowired
    private ImageAnnotator textToImage;

    private AnnotateImageOptions authorOptions;
    private AnnotateImageOptions titleOptions;
    private File2 coversDir;

    @PostConstruct
    private void init() {
        initOptions();
        initCoverImages();
    }

    private String getCoverFileName(int num) {
        return "cover" + num + ".png";
    }

    private void initCoverImages() {
        coversDir = FileUtils.createTempDirectory("book-cover", "");
        for (int i = 1; i <= COVERS_COUNT; i++) {
            String fileName = getCoverFileName(i);
            ClassLoaderUtils.streamSourceForResource(BookCoverCreater.class, fileName)
                    .readTo(coversDir.child(fileName));
        }
    }

    private void initOptions() {
        authorOptions = new AnnotateImageOptions(font, 0, 40, "white", 16);
        titleOptions = new AnnotateImageOptions(font, 0, 140, "white", 20);
    }

    @PreDestroy
    private void destroy() {
        coversDir.deleteRecursiveQuietly();
    }

    private File2 getCoverFile(int num) {
        return coversDir.child(getCoverFileName(num));
    }

    private AnnotateImageOptions prepareOptions(AnnotateImageOptions options, int num) {
        if (COVERS_WITH_BLACK_FONT.containsTs(num)) {
            return options.withImFgColor("black");
        }
        return options;
    }

    private int generateNumOfCoverImage() {
        return Random2.R.nextInt(COVERS_COUNT - 1) + 1;
    }

    private void validateAuthorAndTitle(String author, String title) {
        Validate.isFalse(StringUtils.isBlank(author) && StringUtils.isBlank(title),
                "E-book preview requires book to have cover image, author or title");
    }

    public File2 createCover(String author, String title) {
        validateAuthorAndTitle(author, title);

        int numOfCover = generateNumOfCoverImage();
        String authorLines = prepareLines(author, TextSplitOptions.AUTHOR);
        String titleLines = prepareLines(title.toUpperCase(), TextSplitOptions.TITLE);
        AnnotateImageOptions authorOpts = prepareOptions(authorOptions, numOfCover);
        AnnotateImageOptions titleOpts = prepareOptions(titleOptions, numOfCover);

        return textToImage.convert(
                getCoverFile(numOfCover),
                Tuple2List.fromPairs(authorLines, authorOpts, titleLines, titleOpts),
                ImageFormat.JPEG);
    }

    private String prepareLines(String text, TextSplitOptions textSplitOptions) {
        return StringUtils.join(TextSplitter.splitLongLines(text, textSplitOptions), "\n");
    }
}
