package ru.yandex.chemodan.app.docviewer.convert.djvu;

import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.utils.ExecUtils2;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Validate;

/**
 * @author Vsevolod Tolstopyatov (qwwdfsad)
 */
public class DjvuConverter extends AbstractConverter {

    @Value("${djvu.quality}")
    private int quality;

    @Value("${djvu.subsample}")
    private int subsample;

    @Override
    protected ConvertResultInfo doConvert(
            InputStreamSource input, String contentType,
            TargetType targetType, OutputStreamSource result,
            Option<String> password)
    {
        FileUtils.withFile(
                input,
                (sourceFile) ->
                        FileUtils.withFile(result,
                                (File2 targetFile) ->
                                        convert(sourceFile, targetFile, targetType, quality)));

        return ConvertResultInfo.builder().type(ConvertResultType.PDF).build();
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType.isHtml()
                || convertTargetType == TargetType.PDF
                || convertTargetType == TargetType.PREVIEW;
    }


    private void convert(File2 from, File2 to, TargetType targetType, int quality) {
        Validate.isTrue(25 <= quality && quality <= 150, "Wrong quality param for djvulibre");
        ListF<String> args = Cf.arrayList();
        args.add("ddjvu");
        args.add("-format=pdf");
        args.add("-subsample=" + subsample);
        args.add("-quality=" + quality);
        if (targetType == TargetType.PREVIEW) {
            args.add("-page=1");
        }
        args.add(from.getAbsolutePath());
        args.add(to.getAbsolutePath());
        ExecResult result = ExecUtils2.runScript(args, false);
        if (result.getCode() != 0) {
            throw new RuntimeException("Unexpected exit code in djvulibre: " + result.getCode());
        }
    }

}
