package ru.yandex.chemodan.app.docviewer.convert.doc;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.PostConstruct;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.poi.EncryptedDocumentException;
import org.apache.poi.hwpf.HWPFDocument;
import org.apache.poi.hwpf.HWPFDocumentCore;
import org.apache.poi.hwpf.HWPFOldDocument;
import org.apache.poi.hwpf.converter.AbstractWordUtils;
import org.apache.poi.hwpf.converter.FontReplacer;
import org.apache.poi.hwpf.converter.PicturesManager;
import org.apache.poi.hwpf.converter.WordToHtmlConverter;
import org.apache.poi.hwpf.extractor.Word6Extractor;
import org.apache.poi.hwpf.extractor.WordExtractor;
import org.apache.poi.hwpf.usermodel.PictureType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.w3c.dom.Document;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.adapters.batik.BatikAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.freehep.FreeHepAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageMagickAdapter2;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.states.FileTooBigUserException;
import ru.yandex.chemodan.app.docviewer.states.PasswordProtectedException;
import ru.yandex.chemodan.app.docviewer.utils.ByteArrayOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ImageFileList;
import ru.yandex.chemodan.app.docviewer.utils.OleUtils;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author vlsergey
 */
public class DocConverter extends AbstractConverter {
    private static final Logger logger = LoggerFactory.getLogger(DocConverter.class);

    public final DynamicProperty<Long> docMaxSize =
            new DynamicProperty<>("docviewer.doc.max-size", DataSize.fromMegaBytes(50).toBytes());

    private abstract class AbstractPicturesManager implements PicturesManager {
        protected final Map<ByteArrayHolder, String> imageFiles;
        protected final File temporaryImageDirectory;
        protected final List<Future<?>> workToComplete;

        AbstractPicturesManager(List<Future<?>> workToComplete, File temporaryImageDirectory,
                Map<ByteArrayHolder, String> imageFiles)
        {
            this.workToComplete = workToComplete;
            this.temporaryImageDirectory = temporaryImageDirectory;
            this.imageFiles = imageFiles;
        }

        void addWork(Future<?> docWorkFuture) {
            workToComplete.add(docWorkFuture);
        }

        protected String getStoredFileName(byte[] image) {
            return imageFiles.get(new ByteArrayHolder(image));
        }

        void putStoredFileName(byte[] image, String fileName) {
            imageFiles.put(new ByteArrayHolder(image), fileName);
        }

    }

    private class HtmlPicturesManager extends AbstractPicturesManager {

        HtmlPicturesManager(List<Future<?>> workToComplete, File temporaryImageDirectory,
                Map<ByteArrayHolder, String> imageFiles)
        {
            super(workToComplete, temporaryImageDirectory, imageFiles);
        }

        @Override
        public String savePicture(final byte[] content, final PictureType pictureType,
                final String suggestedName, final float widthInches, final float heightInches)
        {
            ThreadLocalTimeout.check();

            final String stored = getStoredFileName(content);
            if (stored != null) {
                return stored;
            }

            final File2 targetFile;

            final String taskName = "DocConverter-Image-"
                    + uniqueImageConvertTaskKeyCounter.getAndIncrement() + "-" + suggestedName;

            final float vectorWidth = DPI_HTML * widthInches;
            final float vectorHeight = DPI_HTML * heightInches;

            switch (pictureType) {
                case JPEG:
                case TIFF:
                case BMP:
                case PNG:
                    targetFile = new File2(temporaryImageDirectory, suggestedName);
                    addWork(docConvertManager.startImageConversion(taskName,
                            () -> {
                                targetFile.write(content);
                                return null;
                            }, 2));
                    break;
                case EMF:
                    targetFile = new File2(temporaryImageDirectory, suggestedName + ".png");
                    addWork(docConvertManager.startImageConversion(taskName,
                            () -> {
                                ByteArrayOutputStreamSource baoss = new ByteArrayOutputStreamSource();

                                freeHepAdapter.convertEmfToSvg(new ByteArrayInputStreamSource(
                                        content), baoss);
                                batikAdapter.transcodeSvgToPng(new ByteArrayInputStreamSource(
                                        baoss.getByteArray()), new FileOutputStreamSource(
                                        targetFile), vectorWidth, vectorHeight);
                                return null;
                            }, 2));
                    break;
                case WMF:
                    targetFile = new File2(temporaryImageDirectory, suggestedName + ".png");
                    addWork(docConvertManager.startImageConversion(taskName,
                            () -> {
                                ByteArrayOutputStreamSource baoss = new ByteArrayOutputStreamSource();

                                batikAdapter.transcodeWmfToSvg(new ByteArrayInputStreamSource(
                                        content), baoss);
                                batikAdapter.transcodeSvgToPng(new ByteArrayInputStreamSource(
                                        baoss.getByteArray()), new FileOutputStreamSource(
                                        targetFile), vectorWidth, vectorHeight);
                                return null;
                            }, 2));
                    break;
                default:
                    targetFile = new File2(temporaryImageDirectory, suggestedName + ".jpg");
                    addWork(docConvertManager.startImageConversion(taskName,
                            () -> {
                                imageMagickAdapter.convert(
                                        new ByteArrayInputStreamSource(content),
                                        targetFile.asOutputStreamTool(), Option.empty());
                                return null;
                            }, 2));
                    break;
            }

            final String targetUrl = targetFile.getName();
            putStoredFileName(content, targetUrl);
            return targetUrl;
        }
    }

    private static final UrlInputStreamSource BLANK_SVG = new UrlInputStreamSource(
            DocConverter.class.getResource("blank.svg"));

    private static final int DPI_HTML = 96;
    private static final int DPI_FO = 300;

    private static final AtomicLong uniqueImageConvertTaskKeyCounter = new AtomicLong(0L);

    static HWPFDocumentCore loadDoc(InputStreamSource source, final Option<String> password) {
        return source.readX(a -> {
            try {
                // TODO as for 2013-06-11, POI, even in 3.9, does not seem to support passwords for .doc:
                // http://poi.apache.org/encryption.html
                // http://grepcode.com/file/repo1.maven.org/maven2/org.apache.poi/poi-scratchpad/3.8/org/apache/poi/hwpf/model/FileInformationBlock.java#75
                // http://grepcode.com/file/repo1.maven.org/maven2/org.apache.poi/poi-scratchpad/3.9/org/apache/poi/hwpf/model/FileInformationBlock.java#75
                return AbstractWordUtils.loadDoc(a);
            } catch (EncryptedDocumentException exc) {
                throw PasswordProtectedException.atConvertUnsupported(exc);
            } catch (IOException exc) {
                throw IoUtils.translate(exc);
            }
        });
    }

    @Autowired
    private BatikAdapter batikAdapter;

    @Autowired
    private DocImageConvertManager docConvertManager;

    @Autowired
    private ConvertToHtmlHelper convertToHtmlHelper;

    private DocumentBuilderFactory documentBuilderFactory;

    private final FontReplacer fontReplacer = new ExtendedFontReplacer();

    @Autowired
    private FreeHepAdapter freeHepAdapter;

    @Autowired
    private ImageMagickAdapter2 imageMagickAdapter;

    @PostConstruct
    public void afterPropertiesSet() {
        documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true);
    }

    @Override
    public ConvertResultInfo doConvert(InputStreamSource source, final String contentType,
            TargetType convertTargetType, OutputStreamSource result, Option<String> password)
    {
        try {
            final Document document = documentBuilderFactory.newDocumentBuilder().newDocument();
            return doConvert(source, convertTargetType, document, result, password);
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    ConvertResultInfo doConvert(InputStreamSource source, TargetType convertTargetType,
            final Document document, OutputStreamSource result, Option<String> password)
    {
        ThreadLocalTimeout.check();
        source.getFileO().filter(f -> f.length() > docMaxSize.get()).ifPresent(file -> {
            throw FileTooBigUserException.builder()
                    .maxLength(docMaxSize.get())
                    .actualLength(Option.of(file.length()))
                    .mimeType(MimeTypes.MIME_MICROSOFT_WORD)
                    .build();
        });
        try {
            final HWPFDocumentCore wordDocument = loadDoc(source, password);

            if (wordDocument instanceof HWPFOldDocument) {
                throw new UnsupportedOperationException(
                        "DocConverter doesn't support Word 6 files yet");
            }

            DocumentProperties properties = OleUtils.getDocumentProperties(wordDocument.getSummaryInformation());

            switch (convertTargetType) {
                case HTML_ONLY: {
                    WordToHtmlConverter wordToHtmlConverter = new WordToHtmlConverter(document);
                    wordToHtmlConverter.processDocument(wordDocument);

                    int pages = convertToHtmlHelper.splitAndPack(
                            XmlUtils2.convertToDom4j(document), result, false,
                            Option.empty(), false);
                    return ConvertResultInfo.builder().type(ConvertResultType.ZIPPED_HTML).pages(pages)
                            .properties(properties).build();
                }
                case HTML_WITH_IMAGES:
                case HTML_WITH_IMAGES_FOR_MOBILE: {
                    final List<Future<?>> workToComplete = new ArrayList<>();
                    final Map<ByteArrayHolder, String> imageFiles = new HashMap<>();
                    final File2 temporaryImageDirectory = FileUtils.createTempDirectory("htmlimages", ".tmp");
                    try {
                        WordToHtmlConverter wordToHtmlConverter = new WordToHtmlConverter(document);
                        wordToHtmlConverter.setPicturesManager(new HtmlPicturesManager(
                                workToComplete, temporaryImageDirectory.getFile(), imageFiles));
                        wordToHtmlConverter.processDocument(wordDocument);

                        for (Future<?> docWorkFuture : workToComplete) {
                            try {
                                docWorkFuture.get();
                            } catch (Exception exc) {
                                logger.warn("Work didn't completed normally: " + exc, exc);
                            }
                        }

                        FileList images = new FileList(temporaryImageDirectory);
                        int pages = convertToHtmlHelper.splitAndPack(
                                XmlUtils2.convertToDom4j(document), result, false,
                                Option.of(new FileList(temporaryImageDirectory)), false);
                        return ConvertResultInfo.builder()
                                .type(ConvertResultType.ZIPPED_HTML).pages(pages).properties(properties)
                                .images(Option.of(images))
                                .build();
                    } finally {
                        ImageFileList.deleteIfEmpty(temporaryImageDirectory);
                    }
                }
                case PLAIN_TEXT: {
                    String text;
                    if (wordDocument instanceof HWPFOldDocument) {
                        Word6Extractor extractor = new Word6Extractor(
                                (HWPFOldDocument) wordDocument);
                        text = extractor.getText();
                    } else {
                        WordExtractor extractor = new WordExtractor((HWPFDocument) wordDocument);
                        text = extractor.getText();
                    }
                    result.write(text);
                    return ConvertResultInfo.builder().type(ConvertResultType.PLAIN_TEXT).properties(properties)
                            .build();
                }
                default:
                    throw new UnsupportedOperationException("NYI: " + convertTargetType);
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType.isHtml()
                || convertTargetType == TargetType.PLAIN_TEXT;
    }

}
