package ru.yandex.chemodan.app.docviewer.convert.epub;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.XPath;
import org.dom4j.xpath.DefaultXPath;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.book.BookCoverCreater;
import ru.yandex.chemodan.app.docviewer.convert.fb2.Fb2Converter;
import ru.yandex.chemodan.app.docviewer.convert.imagemagick.ImageMagickConverter;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ImageFileList;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.html.AbstractDom4jVisitor;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.utils.html.HyperlinkProcessor;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * TODO fix file extensions by content-type, like in {@link Fb2Converter} (fixImageExtension):
 * <ul>
 * <li>remove default value of <code>fb2.fix.image.extensions</code> property</li>
 * <li>move it to properties file as <code>ebook.fix.image.extensions</code> property</li>
 * <li>walk through items and create mapping from old to fixed hrefs</li>
 * </ul>
 *
 * @author akirakozov
 * @author ssytnik
 */
public class EpubConverter extends AbstractConverter {
    private static final String META_CONTAINTER_FILE = "META-INF/container.xml";

    @Autowired
    private ConvertToHtmlHelper convertToHtmlHelper;

    @Autowired
    private ImageMagickConverter imageMagickConverter;

    @Autowired
    private BookCoverCreater bookCoverCreater;


    private static class LocalUrlVisitor extends AbstractDom4jVisitor {
        private static SetF<String> files = Cf.hashSet();
        private final Option<File2> parentDir;
        private final MapF<String, String> localHtmlToIdMap;

        public LocalUrlVisitor(Option<File2> parentDir,
                MapF<String, String> localHtmlToIdMap)
        {
            this.parentDir = parentDir;
            this.localHtmlToIdMap = localHtmlToIdMap;
        }

        @Override
        protected void visit(Element element) {
            if (StringUtils.equalsIgnoreCase("img", element.getName())) {
                String srcUrl = element.attributeValue("src");
                if (srcUrl != null && HyperlinkProcessor.LOCAL_FILE_PATTERN.matcher(srcUrl).matches()) {
                    // XXX the same file names could be in different folders
                    element.attribute("src").setText(new File2(srcUrl).getName());
                    files.add(createFullPath(parentDir, srcUrl));
                }
            }

            if (StringUtils.equalsIgnoreCase("a", element.getName())) {
                String srcUrl = element.attributeValue("href");

                if (StringUtils.isEmpty(srcUrl)) {
                    return;
                }

                if (HyperlinkProcessor.LOCAL_FILE_PATTERN.matcher(srcUrl).matches()) {
                    String path = createFullPath(parentDir, srcUrl);
                    if (localHtmlToIdMap.containsKeyTs(path)) {
                        element.addAttribute("href", "#" + localHtmlToIdMap.getTs(path));
                    }
                } else if (srcUrl.contains("#")) {
                    String pathPart = StringUtils.substringBefore(srcUrl, "#");
                    if (HyperlinkProcessor.LOCAL_FILE_PATTERN.matcher(pathPart).matches()) {
                        String path = createFullPath(parentDir, StringUtils.substringBefore(srcUrl, "#"));
                        if (localHtmlToIdMap.containsKeyTs(path)) {
                            // XXX could be collision, if different htmls contain tags with the same id
                            element.addAttribute("href", StringUtils.substringAfter(srcUrl, pathPart));
                        }
                    }
                }
            }

        }

        public SetF<String> getImageFiles() {
            return files;
        }
    }


    public static class OpfFileInfo {
        private final Document document;
        private final Option<File2> folder;

        public OpfFileInfo(Document document, Option<File2> folder) {
            this.document = document;
            this.folder = folder;
        }

        public Document getDocument() {
            return document;
        }

        public Option<File2> getFolder() {
            return folder;
        }

        public static OpfFileInfo create(ZipFile epubFile) {
            String opfFilename = getSingleAttributeValue(
                    readXmlEntry(epubFile, META_CONTAINTER_FILE),
                    "/x:container/x:rootfiles/x:rootfile/@full-path");

            return new OpfFileInfo(
                    readXmlEntry(epubFile, opfFilename),
                    new File2(opfFilename).getParent());
        }
    }



    private static String createFullPath(Option<File2> parentDir, String filePath) {
        return parentDir.isPresent() ?
                parentDir.get().child(filePath).getPath() : filePath;
    }

    private static Function<String, String> createFullPathF(final Option<File2> parentDir) {
        return a -> createFullPath(parentDir, a);
    }


    private static XPath xPath(Document document, String query) {
        XPath res = new DefaultXPath(query);
        res.setNamespaceURIs(Cf.map(
                "x", document.getRootElement().getNamespaceURI(),
                "dc", "http://purl.org/dc/elements/1.1/" // for .opf
                ));
        return res;
    }

    private static Option<String> getSingleAttributeValueO(Document document, String query) {
        Attribute attr = (Attribute) xPath(document, query).selectSingleNode(document);
        return attr == null ? Option.empty() : Option.of(attr.getValue());
    }

    private static String getSingleAttributeValue(Document document, String query) {
        return getSingleAttributeValueO(document, query).get();
    }

    @SuppressWarnings("unchecked")
    private static <T extends Node> ListF<T> getNodes(Document document, String query) {
        return Cf.x(xPath(document, query).selectNodes(document));
    }

    private static Document readXmlEntry(ZipFile file, String entryName) {
        return Dom4jUtils.read(new ZipEntryInputStreamSource(file, entryName));
    }


    static ListF<String> getOrderedParts(final OpfFileInfo opfInfo) {
        ListF<Element> items = getNodes(opfInfo.getDocument(),
                "/x:package/x:manifest/x:item[contains(@media-type,'application/xhtml+xml')]");

        final MapF<String, String> idToHrefMap = items.toMap(
                XmlUtils2.attributeValueF("id"), XmlUtils2.attributeValueF("href"));

        ListF<Attribute> idrefs = getNodes(opfInfo.getDocument(), "/x:package/x:spine/x:itemref/@idref");

        return idrefs.map(XmlUtils2.getTextF())
                .map(idToHrefMap.asFunction()).map(createFullPathF(opfInfo.getFolder()));
    }

    private static Option<String> getCoverImage(final OpfFileInfo opfInfo) {
        return getSingleAttributeValueO(opfInfo.getDocument(),
            "/x:package/x:manifest/x:item" +
            "[@id=/x:package/x:metadata/x:meta[@name='cover']/@content]/@href"
        ).map(createFullPathF(opfInfo.getFolder()));
    }


    @SuppressWarnings("unchecked")
    private void mergeParts(OutputStreamSource result, ZipFile epubFile,
            ListF<String> parts, SetF<String> usedImages)
    {
        OutputStreamWriter out = null;
        try {
            out = new OutputStreamWriter(result.getOutput(), CharsetUtils.UTF8_CHARSET);

            MapF<String, String> localHtmlToIdMap = parts.toMapMappingToValue(a -> Random2.R.nextAlnum(5));

            out.write("<html><head><meta http-equiv=\"content-type\" content=\"text/html; charset=UTF-8\"/></head><body>");
            for (String part : parts) {
                Document doc = Dom4jUtils.read(new ZipEntryInputStreamSource(epubFile, part));
                LocalUrlVisitor visitor = new LocalUrlVisitor(
                        new File2(part).getParent(), localHtmlToIdMap);
                visitor.visit(doc);
                usedImages.addAll(visitor.getImageFiles());
                Element body = doc.getRootElement().element("body");
                out.write(("<span id=\"" + localHtmlToIdMap.getTs(part) + "\"></span>"));
                for (Element element : (List<Element>) body.elements()) {
                    out.write(element.asXML());
                }
            }
            out.write("</body></html>");
        } catch (IOException e) {
            throw IoUtils.translate(e);
        } finally {
            IoUtils.closeQuietly(out);
        }
    }

    private static void extractImagesToTempDir(ZipFile epubFile, SetF<String> usedImages, File2 tempImagesDir) {
        for (String img : usedImages) {
            ZipEntry entry = epubFile.getEntry(img);
            if (entry != null) {
                File2 file = tempImagesDir.child(new File2(img).getName());
                new ZipEntryInputStreamSource(epubFile, entry).readTo(file);
            }
        }
    }


    private ConvertResultInfo doConvertToHtml(final ZipFile epubFile,
            final TargetType convertTargetType, final OutputStreamSource result)
    {
        final ListF<String> parts = getOrderedParts(OpfFileInfo.create(epubFile));

        return FileUtils.withEmptyTemporaryFile("epub", ".html", resultCommonHtml -> {
            SetF<String> usedImages = Cf.hashSet();
            mergeParts(resultCommonHtml.asOutputStreamTool(), epubFile, parts, usedImages);

            final File2 tmpImageDir = FileUtils.createTempDirectory("htmlimages", ".tmp");
            Option<FileList> images = Option.empty();
            try {
                if (convertTargetType.isHtmlWithImages()) {
                    extractImagesToTempDir(epubFile, usedImages, tmpImageDir);
                    images = Option.of(new FileList(tmpImageDir));
                }

                Document htmlDocument = Dom4jUtils.read(resultCommonHtml);
                int pages = convertToHtmlHelper.splitAndPack(htmlDocument,
                        result, true, images, true);

                return ConvertResultInfo.builder().type(ConvertResultType.ZIPPED_HTML).pages(pages).images(images).build();
            } finally {
                ImageFileList.deleteIfEmpty(images);
            }
        });
    }

    private ConvertResultInfo doConvertToPreview(ZipFile epubFile, final OutputStreamSource result) {
        OpfFileInfo opfInfo = OpfFileInfo.create(epubFile);
        final Option<String> imageO = getCoverImage(opfInfo);

        if (imageO.isEmpty()) {
            ListF<Element> creatorO = getNodes(opfInfo.getDocument(), "/x:package/x:metadata/dc:creator");
            String author = StringUtils.join(creatorO.map(XmlUtils2.getTextF()), "\n");

            ListF<Element> titleO = getNodes(opfInfo.getDocument(), "/x:package/x:metadata/dc:title");
            // replaceFirstF(" -\\d+$", "") is a hack fix of " -0" in the end of title,
            // see https://jira.yandex-team.ru/browse/DOCVIEWER-1366
            String title = StringUtils.join(
                    titleO.map(XmlUtils2.getTextF()).map(StringUtils.replaceFirstF(" -\\d+$", "")), "\n");

            File2 coverFile = bookCoverCreater.createCover(author, title);
            try {
                return imageMagickConverter.convertImage(
                        coverFile, ImageFormat.JPEG.getContentType(),
                        TargetType.PREVIEW, result, Option.empty());
            } finally {
                coverFile.deleteRecursiveQuietly();
            }

        } else {
            InputStreamSource imageSource = new ZipEntryInputStreamSource(epubFile, imageO.get());

            // TODO need to get content-type from item, and fix extension by it.
            // See this class javadoc, this is a part of that task.
            String contentType = MimeTypes.IMAGE_EXT_2_MIME.getOrElse(
                    new File2(imageO.get()).getExtension(), MimeTypes.IMAGE_X_IMAGEMAGICK_SUPPORTED);

            return imageMagickConverter.convertImage(
                    imageSource, contentType, TargetType.PREVIEW, result, Option.empty());
        }
    }

    @Override
    public ConvertResultInfo doConvert(InputStreamSource input,
            String contentType, final TargetType convertTargetType,
            final OutputStreamSource result, Option<String> password)
    {
        return FileUtils.withZipFile(input, epubFile -> {

            if (convertTargetType == TargetType.PREVIEW) {
                return doConvertToPreview(epubFile, result);
            } else {
                return doConvertToHtml(epubFile, convertTargetType, result);
            }

        });
    }

    @Override
    public boolean isSupported(TargetType targetType) {
        return targetType.isHtml() || targetType == TargetType.PREVIEW;
    }

}
