package ru.yandex.chemodan.app.docviewer.convert.fb2;

import java.util.List;

import javax.xml.transform.Transformer;

import org.apache.commons.lang3.StringUtils;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.XPath;
import org.dom4j.xpath.DefaultXPath;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.book.BookCoverCreater;
import ru.yandex.chemodan.app.docviewer.convert.imagemagick.ImageMagickConverter;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ImageFileList;
import ru.yandex.chemodan.app.docviewer.utils.XslUtils;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author ssytnik
 */
public class Fb2Converter extends AbstractConverter {
    private static final Transformer XSL = XslUtils.getTransformer(Fb2Converter.class, "fb2.xsl");

    @Autowired
    private ConvertToHtmlHelper convertToHtmlHelper;

    @Autowired
    private ImageMagickConverter imageMagickConverter;

    @Autowired
    private BookCoverCreater bookCoverCreater;

    @Value("${fb2.fix.image.extensions}")
    private boolean fixImageExtensions;

    @Override
    public ConvertResultInfo doConvert(
            InputStreamSource input, String contentType, TargetType convertTargetType,
            OutputStreamSource result, Option<String> password)
    {
        Document source = Dom4jUtils.read(input);

        if (convertTargetType == TargetType.PREVIEW) {
            return doConvertToPreview(source, result);
        } else {
            return doConvertToHtml(source, convertTargetType, result);
        }
    }

    private ConvertResultInfo doConvertToPreview(Document source, OutputStreamSource result) {
        Element titleInfo = source.getRootElement().element("description").element("title-info");
        Element coverpage = titleInfo.element("coverpage");

        if (coverpage == null) {
            String authorName;

            Element author = titleInfo.element("author");
            String firstName = author.elementText("first-name");
            String lastName = author.elementText("last-name");

            if (firstName != null && lastName != null) {
                ListF<String> names = Cf.arrayList(firstName, lastName);
//                String middleName = author.elementText("middle-name");
//                if (middleName != null) {
//                    names.add(1, middleName);
//                }
                authorName = names.mkString(" ");
            } else {
                authorName = author.elementText("nickname");
                Check.notNull(authorName);
            }

            String bookTitle = titleInfo.elementText("book-title");

            File2 coverFile = bookCoverCreater.createCover(authorName, bookTitle);

            try {
                return imageMagickConverter.convertImage(
                        coverFile, ImageFormat.JPEG.getContentType(),
                        TargetType.PREVIEW, result, Option.empty());
            } finally {
                coverFile.deleteRecursiveQuietly();
            }
        } else {
            Element firstImage = coverpage.element("image");
            String href = firstImage.attributeValue("href");
            Check.equals('#', href.charAt(0));
            String id = href.substring(1);

            String query = "/fb:FictionBook/fb:binary[@id = '" + id + "']";
            Element binary = (Element) xPath(query).selectSingleNode(source);

            InputStreamSource imageSource = InputStreamSourceUtils.bytes(
                    FastBase64Coder.decode(StringUtils.deleteWhitespace(binary.getText())));
            String contentType = binary.attributeValue("content-type");

            return imageMagickConverter.convertImage(
                    imageSource, contentType, TargetType.PREVIEW, result, Option.empty());
        }
    }

    private ConvertResultInfo doConvertToHtml(Document source,
            TargetType convertTargetType, OutputStreamSource result)
    {
        File2 filesDir = FileUtils.createTempDirectory("fb2-files", ".tmp");
        try {
            Option<FileList> images = Option.empty();
            if (convertTargetType.isHtmlWithImages()) {
                extractFiles(source, filesDir);
                images = Option.of(new FileList(filesDir));
            }

            Document resDoc = XslUtils.transform(XSL, source);
            int pages = convertToHtmlHelper.splitAndPack(resDoc, result, false, images, true);
            return ConvertResultInfo.builder().type(ConvertResultType.ZIPPED_HTML).pages(pages).images(images).build();
        } finally {
            ImageFileList.deleteIfEmpty(filesDir);
        }
    }

    @SuppressWarnings("unchecked")
    private void extractFiles(Document doc, File2 dir) {
        for (Element binary : (List<Element>) doc.getRootElement().elements("binary")) {
            if (fixImageExtensions) {
                fixImageExtension(doc, binary);
            }

            new FileOutputStreamSource(dir.child(binary.attributeValue("id"))).write(
                    FastBase64Coder.decode(StringUtils.deleteWhitespace(binary.getText())));
        }
    }

    @SuppressWarnings("unchecked")
    private void fixImageExtension(Document doc, Element binary) {
        String contentType = binary.attributeValue("content-type");

        if (MimeTypes.IMAGE_MIME_2_EXT.containsKeyTs(contentType)) {
            String actualId = binary.attributeValue("id");
            String expectedId = File2.replaceExtensionInPath(
                    actualId, MimeTypes.IMAGE_MIME_2_EXT.getTs(contentType));

            if (! expectedId.equals(actualId)) {
                binary.addAttribute("id", expectedId);
                String query = "//fb:image/@xlink:href[. = '#" + actualId + "']";
                for (Attribute href : (List<Attribute>) xPath(query).selectNodes(doc))
                {
                    href.setValue("#" + expectedId);
                }
            }
        }
    }

    private static XPath xPath(String query) {
        XPath res = new DefaultXPath(query);
        res.setNamespaceURIs(Cf.map(
                "fb", "http://www.gribuser.ru/xml/fictionbook/2.0",
                "xlink", "http://www.w3.org/1999/xlink"));
        return res;
    }


    @Override
    public boolean isSupported(TargetType targetType) {
        return targetType.isHtml() || targetType == TargetType.PREVIEW;
    }
}
