package ru.yandex.chemodan.app.docviewer.convert.openoffice;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import com.artofsolving.jodconverter.DocumentFormat;
import com.artofsolving.jodconverter.DocumentFormatRegistry;
import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeException;
import com.artofsolving.jodconverter.openoffice.converter.OpenOfficeDocumentConverter;
import com.sun.star.frame.XStorable;
import com.sun.star.lang.XComponent;
import com.sun.star.sheet.XSpreadsheetDocument;
import com.sun.star.task.ErrorCodeIOException;
import com.sun.star.ucb.XFileIdentifierConverter;
import com.sun.star.uno.UnoRuntime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.adapters.openoffice.RetrieableOpenOfficeException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Validate;

/**
 * @author vlsergey
 * @author akirakozov
 */
public class ExtendedOpenOfficeDocumentConverter extends OpenOfficeDocumentConverter {

    private static final Logger logger = LoggerFactory.getLogger(ExtendedOpenOfficeDocumentConverter.class);
    private final Duration closeDocumentTimeout;
    private final Option<String> password;

    public ExtendedOpenOfficeDocumentConverter(
            OpenOfficeConnection connection,
            DocumentFormatRegistry formatRegistry,
            Duration closeDocumentTimeout,
            Option<String> password)
    {
        super(connection, formatRegistry);
        this.closeDocumentTimeout = closeDocumentTimeout;
        this.password = password;

        setDefaultLoadProperty("Hidden", Boolean.TRUE);
        setDefaultLoadProperty("MacroExecutionMode", 0);
        setDefaultLoadProperty("Preview", Boolean.FALSE);
        setDefaultLoadProperty("ReadOnly", Boolean.TRUE);
        setDefaultLoadProperty("UpdateDocMode", 0);
        setDefaultLoadProperty("LinkUpdateModes", 0);
    }

    private void validateFormats(DocumentFormat inputFormat, DocumentFormat outputFormat) {
        Validate.isTrue(inputFormat.isImportable(), "unsupported input format: " + inputFormat.getName());
        Validate.isTrue(inputFormat.isExportableTo(outputFormat),
                "unsupported conversion: from " + inputFormat.getName() + " to " + outputFormat.getName());
    }


    @Override
    public void convert(File inputFile, DocumentFormat inputFormat, File outputFile,
            DocumentFormat outputFormat)
    {
        Validate.isTrue(inputFile.exists(), "inputFile doesn't exist: " + inputFile);
        validateFormats(inputFormat, outputFormat);

        try {
            Map<String, Object> loadProperties = new HashMap<>();
            loadProperties.putAll(getDefaultLoadProperties());
            loadProperties.putAll(inputFormat.getImportOptions());
            if (password.isPresent()) {
                loadProperties.put("Password", password.get());
            }

            Map<String, Object> storeProperties = outputFormat.getExportOptions(inputFormat.getFamily());

            synchronized (openOfficeConnection) {
                XFileIdentifierConverter fileContentProvider = openOfficeConnection.getFileContentProvider();
                String inputUrl = fileContentProvider.getFileURLFromSystemPath("", inputFile.getAbsolutePath());
                String outputUrl = fileContentProvider.getFileURLFromSystemPath("", outputFile.getAbsolutePath());

                logger.debug("convertInternal(...): Using {} as input URL", inputUrl);
                logger.debug("convertInternal(...): Using {} as output URL", outputUrl);

                if (inputFormat.getMimeType().equals(outputFormat.getMimeType())) {
                    IoUtils.copy(new File2(inputFile).getInput(),
                            new File2(outputFile).asAppendOutputStreamTool().getOutput());
                } else {
                    loadAndExport(inputUrl, loadProperties, outputUrl, storeProperties);
                }
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    private void loadAndExport(String inputUri, Map<String, Object> loadProperties,
            String outputUrl, Map<String, Object> storeProperties)
    {
        XComponent xComponent;

        try {
            logger.debug("Loading document...");
            xComponent = OpenOfficeUtils.loadDocument(openOfficeConnection, inputUri, loadProperties);
        } catch (Exception e) {
            throw new OpenOfficeException("conversion failed: could not load input document", e);
        }
        if (xComponent == null) {
            throw new RetrieableOpenOfficeException("conversion failed: input document is null: " + inputUri);
        }

        try {
            if (UnoRuntime.queryInterface(XSpreadsheetDocument.class, xComponent) != null) {
                StylesheetUtils.setPagePropertiesForStylesheet(xComponent);
            }

            try {
                XStorable xStorable = UnoRuntime.queryInterface(XStorable.class, xComponent);
                if (xStorable == null) {
                    throw new OpenOfficeException("conversion failed: storable interface is null");
                }
                logger.debug("Refreshing document...");
                refreshDocument(xComponent);
                logger.debug("Saving document... {}", storeProperties);
                xStorable.storeToURL(outputUrl, toPropertyValues(storeProperties));
            } catch (ErrorCodeIOException e) {
                throw new OpenOfficeException(
                        "conversion failed: could not save output document; OOo errorCode: " + e.ErrCode, e);
            } catch (OpenOfficeException e) {
                throw e;
            } catch (Exception e) {
                throw new OpenOfficeException("conversion failed: could not save output document", e);
            }
        } finally {
            OpenOfficeUtils.closeQuietlyWithTimeout(xComponent, closeDocumentTimeout);
        }
    }

}
