package ru.yandex.chemodan.app.docviewer.convert.openoffice;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.artofsolving.jodconverter.DocumentFamily;
import com.artofsolving.jodconverter.DocumentFormat;
import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeException;
import lombok.Data;
import org.dom4j.Document;
import org.dom4j.Element;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.adapters.openoffice.ExtendedOpenOfficeConnection;
import ru.yandex.chemodan.app.docviewer.adapters.openoffice.OpenOfficeProcessPool;
import ru.yandex.chemodan.app.docviewer.adapters.openoffice.RetrieableOpenOfficeException;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ImageFileList;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.commune.charset.detection.CharsetDetectionUtils;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoFunction1V;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author vlsergey
 * @author akirakozov
 */
@Data
public class OpenOfficeConverter extends AbstractConverter {
    private static final Logger logger = LoggerFactory.getLogger(OpenOfficeConverter.class);

    private static final Pattern IMAGE_INSIDE_HTML_SRC_ATTRIBUTE_PATTERN = Pattern.compile("data:(.+);base64,(.*)");

    @Autowired
    private ConvertToHtmlHelper convertToHtmlHelper;
    @Autowired
    private OpenOfficeProcessPool commonOOProcessPool;
    @Autowired
    private OpenOfficeProcessPool presentationOOProcessPool;
    @Autowired
    private MimeDetector mimeDetector;

    @Value("${openoffice.spurious.fail.retries}")
    private int spuriousFailRetries;
    @Value("${max.html.result.size}")
    private DataSize maxHtmlResultSize;
    @Value("${oo.max.attempts}")
    private int maxAttemptsNumber;
    @Value("${oo.close.document.timeout}")
    private Duration closeDocumentTimeout;
    @Value("${oo.invalidate.on.fail}")
    private boolean invalidateOnFail;
    @Value("${oo.txt.max.bytes.for.preview}")
    private DataSize maxTxtBytesCountForPreview;
    @Value("${doc.prefer.pdf.target.type.lo}")
    boolean docPreferPdfTargetType;
    @Value("${openoffice.result.tmp.dir}")
    private String resultTmpDir;

    /**
     * @return positive value
     */
    protected int getMaxAttemptsNumber() {
        return maxAttemptsNumber;
    }

    private OpenOfficeProcessPool getOpenOfficeProcessPool(boolean isPresentation) {
        return isPresentation ? presentationOOProcessPool : commonOOProcessPool;
    }

    private boolean invalidateOnFail(boolean isPresentation) {
        return isPresentation || invalidateOnFail;
    }

    @Override
    public ConvertResultInfo doConvert(final InputStreamSource source, String contentType,
            final TargetType convertTargetType, final OutputStreamSource result,
            final Option<String> password)
    {
        OpenOfficeException lastException = null;
        for (int attempt = 1; attempt <= getMaxAttemptsNumber(); attempt++) {
            try {
                return doConvertAttempt(source, contentType, convertTargetType, result, password);
            } catch (RetrieableOpenOfficeException e) {
                logger.debug("Attempt " + attempt + " failed: " + e.getMessage(), e);
                if (Thread.currentThread().isInterrupted()) {
                    throw e;
                } else {
                    lastException = e;
                }
            }
        }
        throw lastException;
    }

    public ConvertResultInfo doConvertAttempt(final InputStreamSource source, String contentType,
            final TargetType convertTargetType0, final OutputStreamSource result,
            final Option<String> password)
    {
        final DocumentFormat inputFormat = ExtendedDocumentFormatRegistry.INSTANCE.getFormatByMimeType(contentType);

        final TargetType convertTargetType = convertTargetType0 == TargetType.HTML_WITH_IMAGES_FOR_MOBILE
                ? TargetType.HTML_WITH_IMAGES
                : (preferPdf(inputFormat)) && convertTargetType0.isHtml() ? TargetType.PDF : convertTargetType0;

        logger.debug("Converting '{}'... ", source);

        Check.notNull(inputFormat, "inputFormat");

        final DocumentFormat outputFormat = getOutputFormat(inputFormat, convertTargetType);
        switch (convertTargetType) {
            case HTML_ONLY:
            case HTML_WITH_IMAGES: {
                final File2 directory = FileUtils.createTempDirectory("result", ".ooc", getOfficeResultDir());
                Option<FileList> images = convertTargetType == TargetType.HTML_ONLY ?
                        Option.empty() : Option.of(new FileList(directory));
                try {
                    File2 resultHtmlFile = new File2(directory, "result.html");

                    doConvert(source, inputFormat, outputFormat,
                            new FileOutputStreamSource(resultHtmlFile),
                            password);

                    DataSize resultHtmlSize = DataSize.fromBytes(resultHtmlFile.length());
                    logger.debug("Result html file size: {}", resultHtmlSize.toPrettyString());
                    if (resultHtmlSize.lt(maxHtmlResultSize)) {
                        Document htmlDocument = XmlUtils2.parseHtmlToDom4j(resultHtmlFile);

                        if (convertTargetType == TargetType.HTML_WITH_IMAGES) {
                            extractFiles(htmlDocument, directory);
                        }

                        int pages = convertToHtmlHelper.splitAndPack(htmlDocument,
                                result, true, images, false);
                        return ConvertResultInfo.builder()
                                .type(ConvertResultType.ZIPPED_HTML).pages(pages).images(images)
                                .build();
                    } else {
                        // hack for DOCVIEWER-674
                        // TODO: special code for this situation?
                        directory.deleteRecursiveQuietly();
                        throw new UserException(ErrorCode.UNKNOWN_CONVERT_ERROR,
                                "Too big result html size: " + resultHtmlSize.toPrettyString());
                    }
                } finally {
                    ImageFileList.deleteIfEmpty(images);
                }
            }
            case PREVIEW:
                if (MimeTypes.MIME_TEXT_PLAIN.equals(inputFormat.getMimeType())) {
                    FileUtils.withEmptyTemporaryFile("tmp", ".txt", tmp -> {
                        tmp.asOutputStreamTool().write(
                                source.readBytesLimited((int) maxTxtBytesCountForPreview.toBytes()));
                        doConvert(tmp, inputFormat, outputFormat, result, password);
                    });
                } else {
                    doConvert(source, inputFormat, outputFormat, result, password);
                }
                return ConvertResultInfo.builder().type(ConvertResultType.PDF).build();
            case PDF:
            case PLAIN_TEXT: {
                doConvert(source, inputFormat, outputFormat, result, password);
                return ConvertResultInfo.builder().type(convertTargetType == TargetType.PLAIN_TEXT ?
                        ConvertResultType.PLAIN_TEXT : ConvertResultType.PDF).build();
            }
            case MICROSOFT_OOXML_OFFICE: {
                doConvert(source, inputFormat, outputFormat, result, password);
                return ConvertResultInfo.builder().type(getConvertResultTypeByMimeType(outputFormat.getMimeType())).build();
            }
            default: {
                throw new UnsupportedOperationException("NYI: " + convertTargetType);
            }
        }
    }

    private boolean preferPdf(DocumentFormat inputFormat) {
        return docPreferPdfTargetType && !DocumentFamily.SPREADSHEET.equals(inputFormat.getFamily());
    }

    @NotNull
    private File2 getOfficeResultDir() {
        return StringUtils.notBlankO(resultTmpDir)
                .map(File2::new)
                .getOrElse(FileUtils.getRootTempDir());
    }

    @SuppressWarnings("unchecked")
    private void extractFiles(Document doc, File2 dir) {
        for (Element image : (List<Element>) doc.selectNodes("//IMG")) {
            Matcher matcher = IMAGE_INSIDE_HTML_SRC_ATTRIBUTE_PATTERN.matcher(image.attributeValue("src"));
            if (matcher.find()) {
                String imageMimeType = MimeDetector.normalize(matcher.group(1));
                String imageExtension = "";
                if (MimeTypes.IMAGE_MIME_2_EXT.containsKeyTs(imageMimeType)) {
                    imageExtension = "." + MimeTypes.IMAGE_MIME_2_EXT.getTs(imageMimeType);
                }

                File2 imageLocalFile = dir.child("image-" + Random2.R.nextAlnum(18) + imageExtension);
                imageLocalFile.write(FastBase64Coder.decode(matcher.group(2)));
                image.addAttribute("src", imageLocalFile.getName());
            }
        }
    }

    // TODO: retrieve document properties
    private void doConvert(final InputStreamSource source, final DocumentFormat inputFormat,
            final DocumentFormat outputFormat, final OutputStreamSource result,
            final Option<String> password)
    {
        FileUtils.withFile(source, new IoFunction1V<File2>() {
            public void applyWithException(final File2 inputFile) {
                boolean isPresentation = inputFormat.getFamily() == DocumentFamily.PRESENTATION;
                ExtendedOpenOfficeConnection connection =
                        getOpenOfficeProcessPool(isPresentation).openConnection(spuriousFailRetries);
                boolean completed = false;
                try {
                    final ExtendedOpenOfficeDocumentConverter converter =
                            new ExtendedOpenOfficeDocumentConverter(
                                    connection,
                                    ExtendedDocumentFormatRegistry.INSTANCE,
                                    closeDocumentTimeout,
                                    password);

                    boolean convertToUtf8 = false;
                    String encoding = "utf-8";
                    if (MimeTypes.MIME_TEXT_PLAIN.equals(inputFormat.getMimeType())) {
                        encoding = CharsetDetectionUtils.detect(inputFile, encoding);
                        convertToUtf8 = !"utf-8".equals(encoding);
                    }

                    final String srcEncoding = convertToUtf8 ? encoding : null;
                    if (convertToUtf8) {
                        FileUtils.withEmptyTemporaryFile("tmp", ".txt", tmp -> {
                            tmp.asOutputStreamTool().write(source.readText(srcEncoding));
                            doConvertInner(tmp, inputFormat, outputFormat, result, converter);
                        });
                    } else {
                        doConvertInner(inputFile, inputFormat, outputFormat, result, converter);
                    }

                    completed = true;
                } finally {
                    logger.debug("Disconnecting quietly, completed={}", completed);
                    connection.disconnectQuietly(!completed && invalidateOnFail(isPresentation));
                }
            }

            private void doConvertInner(final File2 inputFile, final DocumentFormat inputFormat,
                    final DocumentFormat outputFormat, final OutputStreamSource result,
                    final ExtendedOpenOfficeDocumentConverter converter)
            {
                FileUtils.withFile(result, resultFile -> {
                    Stopwatch watch = Stopwatch.createAndStart();
                    converter.convert(inputFile.getFile(), inputFormat, resultFile.getFile(), outputFormat);
                    watch.stopAndLog("Conversion complete", logger);
                });
            }
        });
    }

    private DocumentFormat getOutputFormat(final DocumentFormat inputFormat, final TargetType convertTargetType) {
        ExtendedDocumentFormatRegistry dfr = ExtendedDocumentFormatRegistry.INSTANCE;
        if (convertTargetType.isHtml()) {
            return dfr.getFormatByMimeType(MimeTypes.MIME_HTML);
        } else if (convertTargetType == TargetType.PLAIN_TEXT) {
            return dfr.getFormatByMimeType(MimeTypes.MIME_TEXT_PLAIN);
        } else if (convertTargetType == TargetType.PDF) {
            return dfr.getFormatByMimeType(MimeTypes.MIME_PDF);
        } else if (convertTargetType == TargetType.PREVIEW) {
            return ExtendedDocumentFormatRegistry.exportFirstPdfPageFormat();
        } else if (convertTargetType == TargetType.MICROSOFT_OOXML_OFFICE) {
            return getMsOoxmlOfficeFormatByOriginalOfficeFormat(inputFormat);
        } else {
            throw new UnsupportedOperationException("NYI: " + convertTargetType);
        }
    }

    private DocumentFormat getMsOoxmlOfficeFormatByOriginalOfficeFormat(DocumentFormat original) {
        ExtendedDocumentFormatRegistry dfr = ExtendedDocumentFormatRegistry.INSTANCE;
        String mimeType = original.getMimeType();
        String contentTypeFamily = mimeDetector.getContentTypeFamily(mimeType);
        switch (contentTypeFamily) {
            case "text":
                return dfr.getFormatByMimeType(MimeTypes.MIME_MICROSOFT_OOXML_WORD);
            case "spreadsheet":
                return dfr.getFormatByMimeType(MimeTypes.MIME_MICROSOFT_OOXML_EXCEL);
            case "presentation":
                return dfr.getFormatByMimeType(MimeTypes.MIME_MICROSOFT_OOXML_POWERPOINT);
            default:
                throw new UnsupportedOperationException("Can't obtain appropriate output format by original: " + mimeType);
        }
    }

    private ConvertResultType getConvertResultTypeByMimeType(String mimeType) {
        switch (mimeType) {
            case MimeTypes.MIME_MICROSOFT_OOXML_WORD:
                return ConvertResultType.MICROSOFT_OOXML_WORD;
            case MimeTypes.MIME_MICROSOFT_OOXML_EXCEL:
                return ConvertResultType.MICROSOFT_OOXML_EXCEL;
            case MimeTypes.MIME_MICROSOFT_OOXML_POWERPOINT:
                return ConvertResultType.MICROSOFT_OOXML_POWERPOINT;
            default:
                throw new UnsupportedOperationException("Can't obtain appropriate ConvertResultType by original: " + mimeType);
        }
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType.isHtml()
                || convertTargetType == TargetType.PLAIN_TEXT
                || convertTargetType == TargetType.PDF
                || convertTargetType == TargetType.PREVIEW
                || convertTargetType == TargetType.MICROSOFT_OOXML_OFFICE;
    }

}
