package ru.yandex.chemodan.app.docviewer.convert.openoffice;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import com.sun.star.beans.PropertyValue;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.lang.XComponent;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.util.CloseVetoException;
import com.sun.star.util.XCloseable;
import org.joda.time.Duration;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author vlsergey
 * @author akirakozov
 */
class OpenOfficeUtils {

    private static final Logger logger = LoggerFactory.getLogger(OpenOfficeUtils.class);

    public static void closeQuietlyWithTimeout(final XComponent document, Duration closeDocumentTimeout) {
        if (document == null)
            return;

        ExecutorService executor = Executors.newSingleThreadExecutor();
        Future<Void> future = executor.submit(() -> {
            closeQuietly(document);
            return null;
        });

        try {
            Stopwatch w = Stopwatch.createAndStart();
            future.get(closeDocumentTimeout.getStandardSeconds(), TimeUnit.SECONDS);
            w.stopAndLog("Document closing", logger);
        } catch (TimeoutException e) {
            logger.warn("Couldn't close document, because of timeout");
        } catch (Exception e) {
            logger.warn(e, e);
        }

        executor.shutdownNow();
    }

    private static void closeQuietly(XComponent document) {
        if (document == null)
            return;

        try {
            XCloseable closeable = UnoRuntime.queryInterface(XCloseable.class, document);
            if (closeable != null) {
                try {
                    closeable.close(true);
                } catch (CloseVetoException exc) {
                    logger.warn("document.close() vetoed: " + exc, exc);
                } catch (Throwable exc) {
                    logger.warn("Unable to close document: " + exc, exc);
                }
                return;
            }
        } catch (Throwable exc) {
            logger.warn("UnoRuntime.queryInterface threw " + exc, exc);
        }

        try {
            document.dispose();
        } catch (Throwable exc) {
            logger.warn("document.dispose() threw " + exc, exc);
        }
    }

    public static XComponent loadDocument(OpenOfficeConnection connection, String inputUrl,
            Map<String, Object> loadProperties)
    {
        Map<String, Object> actualLoadProperties = new LinkedHashMap<>(loadProperties);
        if (!actualLoadProperties.containsKey("Password"))
            actualLoadProperties.put("Password", "");

        synchronized (connection) {
            try {
                XComponentLoader desktop = connection.getDesktop();
                XComponent xComponent = desktop.loadComponentFromURL(inputUrl, "_blank", 0,
                        toPropertyValues(actualLoadProperties));

                return xComponent;
            } catch (Exception exc) {
                throw ExceptionUtils.translate(exc);
            }
        }
    }

    public static PropertyValue property(String name, Object value) {
        PropertyValue property = new PropertyValue();
        property.Name = name;
        property.Value = value;
        return property;
    }

    public static PropertyValue[] toPropertyValues(Map<String, Object> properties) {
        PropertyValue[] propertyValues = new PropertyValue[properties.size()];
        int i = 0;
        for (Map.Entry<String, Object> entry : properties.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Map<?, ?>) {
                @SuppressWarnings("unchecked")
                Map<String, Object> subProperties = (Map<String, Object>) value;
                value = toPropertyValues(subProperties);
            }
            propertyValues[i++] = property(entry.getKey(), value);
        }
        return propertyValues;
    }

}
