package ru.yandex.chemodan.app.docviewer.convert.pdf;

import org.apache.pdfbox.exceptions.CryptographyException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.util.PDFTextStripper;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.PopplerAdapter;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.PasswordProtectedException;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.PdfEncodingFixer;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.IoFunction1V;
import ru.yandex.misc.io.OutputStreamSource;

/**
 * @author vlsergey
 */
public class PdfConverter extends AbstractConverter {

    @Autowired
    private PopplerAdapter popplerAdapter;

    @Autowired
    private PdfEncodingFixer pdfEncodingFixer;


    @Override
    public ConvertResultInfo doConvert(final InputStreamSource source, final String contentType,
            final TargetType convertTargetType, final OutputStreamSource result,
            final Option<String> password)
    {
        switch (convertTargetType) {
            case PDF:
                return doConvertToPdf(source, result, password);
            case PREVIEW:
                return doConvertToPdfFirstPage(source, result, password);
            case PLAIN_TEXT:
                return doConvertToText(source, result, password);
            default:
                throw new UserException(ErrorCode.UNSUPPORTED_CONVERTION);
        }
    }

    private ConvertResultInfo doConvertToPdf(InputStreamSource pdfDocumentSource,
            OutputStreamSource result, Option<String> password)
    {
        unencryptDocument(pdfDocumentSource, result, password);

        return ConvertResultInfo.builder().type(ConvertResultType.PDF).build();
    }

    private ConvertResultInfo doConvertToPdfFirstPage(
            final InputStreamSource source, final OutputStreamSource result,
            final Option<String> password)
    {
        FileUtils.withEmptyTemporaryFile("unencrypted", ".pdf", unencryptedPdf -> {
            unencryptDocument(source, unencryptedPdf.asOutputStreamTool(), password);
            popplerAdapter.separate(unencryptedPdf, result, 1, 1);
        });

        return ConvertResultInfo.builder().type(ConvertResultType.PDF).build();
    }

    protected ConvertResultInfo doConvertToText(final InputStreamSource pdfDocumentSource,
            final OutputStreamSource result, final Option<String> password)
    {
        return PdfUtils.withExistingDocument(pdfDocumentSource, true, (IoFunction<PDDocument, ConvertResultInfo>) document -> {
            unencryptIfEncrypted(document, password);
            pdfEncodingFixer.fixFonts(document);

            DocumentProperties properties = PdfUtils.getProperties(document);

            PdfUtils.filterTextOperators(document, PdfUtils.preserveInHtmlHandler(false));

            final PDFTextStripper stripper = new PDFTextStripper("utf-8");
            stripper.setForceParsing(true);
            stripper.setSortByPosition(true);
            stripper.setShouldSeparateByBeads(true);
            stripper.setEndPage(50);

            result.asWriterTool("utf-8").writeBuffered(writer -> stripper.writeText(document, writer));

            return ConvertResultInfo.builder().type(ConvertResultType.PLAIN_TEXT).properties(properties).build();
        });
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType == TargetType.PDF ||
                convertTargetType == TargetType.PREVIEW ||
                convertTargetType == TargetType.PLAIN_TEXT;
    }

    private void unencryptDocument(
            final InputStreamSource pdfDocumentSource, final OutputStreamSource result,
            final Option<String> password)
    {
        PdfUtils.withExistingDocument(pdfDocumentSource, true, (IoFunction1V<PDDocument>) document -> {
            boolean encrypted = unencryptIfEncrypted(document, password);
            if (encrypted) {
                PdfUtils.write(document, result);
            } else {
                pdfDocumentSource.readTo(result);
            }
        });
    }

    static boolean unencryptIfEncrypted(PDDocument document, Option<String> password) {
        try {
            if (document.isEncrypted()) {
                document.decrypt(password.getOrElse(""));
                document.setAllSecurityToBeRemoved(true);
                return true;
            } else {
                return false;
            }
        } catch (CryptographyException exc) {
            throw PasswordProtectedException.atConvert(exc);
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }
}
