package ru.yandex.chemodan.app.docviewer.convert.pdf;

import javax.annotation.PostConstruct;
import javax.xml.parsers.DocumentBuilderFactory;

import org.dom4j.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.result.PageInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlDocumentExtCssFacade;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.thread.ThreadLocalTimeout;

public class PdfToHtml {

    private static final Logger logger = LoggerFactory.getLogger(PdfToHtml.class);

    @Autowired
    private HtmlPostprocessor htmlPostprocessor;

    private DocumentBuilderFactory documentBuilderFactory;

    @PostConstruct
    public void afterPropertiesSet() {
        documentBuilderFactory = DocumentBuilderFactory.newInstance();
    }

    public Document getPageHtml(
            PagesInfo pagesInfo, Option<Integer> oneBasedPageIndex, boolean withImages, DimensionO size)
    {
        try {
            org.w3c.dom.Document document = documentBuilderFactory.newDocumentBuilder().newDocument();
            HtmlDocumentExtCssFacade htmlDocumentFacade = new HtmlDocumentExtCssFacade(document);

            org.w3c.dom.Element metaEncoding = htmlDocumentFacade.getDocument().createElement("META");
            metaEncoding.setAttribute("http-equiv", "content-type");
            metaEncoding.setAttribute("content", "text/html; charset=utf-8");
            htmlDocumentFacade.getHead().appendChild(metaEncoding);

            processPage(pagesInfo, oneBasedPageIndex, withImages, size, htmlDocumentFacade);

            htmlDocumentFacade.updateStylesheet();
            return XmlUtils2.convertToDom4j(htmlDocumentFacade.getDocument());
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }

    }

    private void processPage(final PagesInfo pagesInfo,
            final Option<Integer> oneBasedPageIndex, final boolean withImages,
            final DimensionO size, final HtmlDocumentExtCssFacade htmlDocumentFacade)
    {
            if (oneBasedPageIndex.isPresent()) {
                final int pageIndexInt = oneBasedPageIndex.get();
                Element pageDiv = processPage(
                        pagesInfo.getPageInfos().get(pageIndexInt - 1), pageIndexInt, withImages, size, htmlDocumentFacade);
                if (pageDiv != null) {
                    htmlDocumentFacade.getBody().appendChild(pageDiv);
                }
            } else {
                for (int p = 0; p < pagesInfo.getCount(); p++) {
                    Element pageDiv = processPage(
                            pagesInfo.getPageInfos().get(p), p + 1, withImages, size, htmlDocumentFacade);
                    if (pageDiv != null) {
                        htmlDocumentFacade.getBody().appendChild(pageDiv);
                    }
                }
            }
    }

    private Element processPage(PageInfo pageInfo, int oneBasedPageIndex, boolean withImages,
            final DimensionO size, final HtmlDocumentExtCssFacade htmlDocumentFacade)
    {
        ThreadLocalTimeout.check();

        logger.debug("Processing page #{}", oneBasedPageIndex);
        int zeroBasedPageIndex = oneBasedPageIndex - 1;

        Dimension boundingBox = pageInfo.getBoundingBox(size);

        int htmlWidth = boundingBox.getWidth();
        int htmlHeight = boundingBox.getHeight();

        final Element pageDiv = htmlDocumentFacade.createBlock();
        htmlDocumentFacade.addStyleClass(pageDiv, "p", "position:relative;margin:auto;");
        htmlDocumentFacade.addStyleClass(pageDiv, "pw", "width:" + htmlWidth + "px;");
        htmlDocumentFacade.addStyleClass(pageDiv, "ph", "height:" + htmlHeight + "px;");

        if (withImages) {
            htmlPostprocessor.addPageBackground(htmlDocumentFacade, pageDiv, zeroBasedPageIndex,
                    htmlWidth, htmlHeight);
        }

        return pageDiv;
    }

}
