package ru.yandex.chemodan.app.docviewer.convert.pdf.fonts;

import java.util.EnumMap;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.cache.impl.LruCache;

/**
 * Accepts pdf font name, obtains {@link FontNameInfo} for it,
 * also looks up if there is a {@link KnownFontSourceInfo}
 * (if so, {@literal @}font-face also appears at the result)
 * and constructs overall {@link FontCss} css style info for this font,
 * that can be injected to the output html
 *
 * @author ssytnik
 */
public class FontManager {

    /**
     * Important because font name can be parsed on access to each word
     */
    private static final LruCache<String, FontCss> cache = new LruCache<>(1000);


    private MapF<String, ListF<KnownFontSourceInfo>> knownFonts;
    private MapF<String, String> knownFontsIds;

    public FontManager(Tuple2List<String, KnownFontSourceInfo> knownFonts) {
        this.knownFonts = knownFonts.groupBy1();

        this.knownFontsIds = Cf.hashMap();
        for (String key : this.knownFonts.keySet()) {
            knownFontsIds.put(key, "ff" + knownFontsIds.size());
        }
    }

    public FontCss getFontCss(final String pdfFontName) {
        return cache.getFromCache(pdfFontName, () -> Optional.of(doGetFontCss(pdfFontName))).get();
    }

    private FontCss doGetFontCss(String pdfFontName) {
        Optional<FontNameInfo> fontNameInfoO = FontNameParser.parse(pdfFontName);
        if (!fontNameInfoO.isPresent()) {
            return new FontCss("");
        }

        FontNameInfo fontNameInfo = fontNameInfoO.get();

        ListF<String> cssFamilies = fontNameInfo.getCssFamilies();
        Option<Tuple2<String, String>> fontFaceIdAndStyleO = Option.empty();

        for (String fontFamily : Cf.list(fontNameInfo.getFontFamilyWithStyles(), fontNameInfo.getFontFamily())) {
            if (knownFontsIds.containsKeyTs(fontFamily)) {
                String fontId = knownFontsIds.getTs(fontFamily);
                ListF<KnownFontSourceInfo> fontSources = knownFonts.getTs(fontFamily);

                fontFaceIdAndStyleO = Option.of(Tuple2.tuple(fontId, getFontFaceCss(fontId, fontSources)));
                cssFamilies = Cf.list(fontId).plus(cssFamilies);

                break;
            }
        }

        String fontStyle = Cf.list(getCss("font-family", cssFamilies, true))
                .plus(getFontStyleCss(fontNameInfo.getCssStyles()))
                .mkString("\n");

        return new FontCss(fontStyle, fontFaceIdAndStyleO);
    }


    private String getCss(String name, ListF<String> values, boolean quote) {
        if (quote) {
            values = values.map(quoteF);
        }
        return String.format("%s: %s;", name, values.mkString(", "));
    }

    private ListF<String> getFontStyleCss(EnumMap<CssFontStyleProperty, String> cssStyles) {
        return Cf.x(cssStyles).mapEntries((a, b) -> getCss(a.getCssName(), Cf.list(b), true));
    }

    private String getFontFaceCss(String fontId, ListF<KnownFontSourceInfo> fontSources) {
        ListF<KnownFontSourceInfo> sortedSources = fontSources.sortedBy(KnownFontSourceInfo::getFormat);
        return
                getCss("font-family", Cf.list(fontId), true) + "\n" +
                getCss("src", sortedSources.map(KnownFontSourceInfo::getCssSrcPropertyValue), false);

    }

    private Function<String, String> quoteF = a -> StringUtils.contains(a, ' ') ? "'" + a + "'" : a;

}
