package ru.yandex.chemodan.app.docviewer.convert.pdf.fonts;

import ru.yandex.misc.lang.Validate;

/**
 * @author ssytnik
 */
public class KnownFontSourceInfo {

    public enum FontSource {
        URL_AT_CLIENT,
        BASE64_DATA,
        ;
    }

    // order is important
    public enum FontFormat {
        // Links related to font mime types and css format values:
        //  http://stackoverflow.com/questions/3594823/mime-type-for-woff-fonts
        //  http://blogs.planetcloud.co.uk/mygreatdiscovery/post/Font-MIME-Types.aspx
        //  http://www.xiper.net/collect/html-and-css-tricks/typographics/font-face-in-the-details.html
        //  http://paulirish.com/2009/bulletproof-font-face-implementation-syntax/
        //  http://stackoverflow.com/questions/10086876/how-is-this-font-face-in-css3-embedded-into-a-css-file
        EOT("application/vnd.ms-fontobject", "embedded-opentype"), // mime: application/vnd.ms-fontobject
        WOFF("application/x-font-woff", "woff"), // mime: application/font-woff, font/woff, font/x-woff, font/opentype
        TTF("font/truetype", "truetype"), // mime: font/woff
        OTF("font/opentype", "opentype"),
        SVG("image/svg+xml", "svg"), // mime: image/svg+xml
        ;

        private String mimeType;
        private String cssFormatValue;

        FontFormat(String mimeType, String cssFormatValue) {
            this.mimeType = mimeType;
            this.cssFormatValue = cssFormatValue;
        }

        public String getMimeType() {
            return mimeType;
        }

        public String getCssFormatValue() {
            return cssFormatValue;
        }
    }

    private FontSource source;
    private FontFormat format;
    // content depends on source: url or base64 data
    private String data;


    private KnownFontSourceInfo(FontSource source, FontFormat format, String data) {
        Validate.notNull(source);
        Validate.notNull(format);
        Validate.notNull(data);

        this.source = source;
        this.format = format;
        this.data = data;
    }

    public String getCssSrcPropertyValue() {
        String url;
        switch (source) {
        case URL_AT_CLIENT: {
            url = data + (format == FontFormat.EOT ? "?" : "");
            break;
        }
        case BASE64_DATA: {
            url = String.format("data:%s;charset=utf-8;base64,%s", format.getMimeType(), data);
            break;
        }
        default: {
            throw new IllegalStateException("Unknown font source: " + source);
        }
        }
        return String.format("url('%s') format('%s')", url, format.getCssFormatValue());
    }


    public FontSource getSource() {
        return source;
    }

    public FontFormat getFormat() {
        return format;
    }

    public String getData() {
        return data;
    }

    @Override
    public String toString() {
        return String.format("%s/%s [%s]", format, source, (source == FontSource.URL_AT_CLIENT ? data : "..."));
    }

}
