package ru.yandex.chemodan.app.docviewer.convert.text;

import java.io.IOException;
import java.io.Reader;

import javax.annotation.PostConstruct;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.poi.hwpf.converter.HtmlDocumentFacade;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.w3c.dom.Document;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.adapters.imagemagick.ImageAnnotator;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.imagemagick.ImageMagickConverter;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.commune.charset.detection.CharsetDetectionUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;

/**
 * @author ssytnik
 */
public abstract class TextConverterSupport extends AbstractConverter {
    private static final Logger logger = LoggerFactory.getLogger(TextConverterSupport.class);

    @Autowired
    protected ConvertToHtmlHelper convertToHtmlHelper;

    @Autowired
    private ImageMagickConverter imageMagickConverter;

    @Autowired
    private ImageAnnotator textToImage;

    private DocumentBuilderFactory documentBuilderFactory;


    @PostConstruct
    public void afterPropertiesSet() {
        documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(false);
    }

    @Override
    public ConvertResultInfo doConvert(final InputStreamSource input, final String contentType,
            final TargetType convertTargetType, final OutputStreamSource result,
            final Option<String> password)
    {
        String encoding = CharsetDetectionUtils.detect(input, "utf-8");
        logger.debug("Detected encoding: {}", encoding);

        if (convertTargetType == TargetType.PLAIN_TEXT) {
            return doConvertToText(input, result, encoding);
        } else {
            input.asReaderTool(encoding).asFactory();
            return IoUtils.read(
                    input.asReaderTool(encoding).asFactory(),
                    reader -> ConvertResultInfo.builder()
                            .type(ConvertResultType.ZIPPED_HTML)
                            .pages(doConvertToHtml(reader, result))
                            .build());
        }
    }

    private ConvertResultInfo doConvertToText(
            InputStreamSource input, OutputStreamSource result, String encoding)
    {
        result.asWriterTool().writeLine(input.readText(encoding));
        return ConvertResultInfo.builder().type(ConvertResultType.PLAIN_TEXT).build();
    }

    private int doConvertToHtml(Reader inputReader, OutputStreamSource result) {
        try {
            Document document = documentBuilderFactory.newDocumentBuilder().newDocument();
            HtmlDocumentFacade htmlDocumentFacade = new HtmlDocumentFacade(document);

            doConvertToHtml(inputReader, htmlDocumentFacade);
            htmlDocumentFacade.updateStylesheet();

            return splitAndPackResult(result, document);
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    protected int splitAndPackResult(OutputStreamSource result, Document document) {
        return convertToHtmlHelper.splitAndPack(
                XmlUtils2.convertToDom4j(document), result, false, Option.empty(), true);
    }

    protected abstract void doConvertToHtml(Reader inputReader, HtmlDocumentFacade htmlDocumentFacade)
            throws IOException;

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType.isHtml()
                || convertTargetType == TargetType.PLAIN_TEXT;
    }
}
