package ru.yandex.chemodan.app.docviewer.convert.xls;

import javax.annotation.PostConstruct;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.poi.EncryptedDocumentException;
import org.apache.poi.hssf.converter.ExcelToHtmlConverter;
import org.apache.poi.hssf.extractor.ExcelExtractor;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hwpf.converter.HtmlDocumentFacade;
import org.apache.poi.ss.util.CellRangeAddress;
import org.springframework.beans.factory.annotation.Autowired;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.AbstractConverter;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.doc.ExtendedFontReplacer;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.states.FileTooBigUserException;
import ru.yandex.chemodan.app.docviewer.states.PasswordProtectedException;
import ru.yandex.chemodan.app.docviewer.utils.OleUtils;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlDocumentExtCssFacade;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author vlsergey
 */
public class XlsConverter extends AbstractConverter {

    public final DynamicProperty<Long> xlsMaxSize =
            new DynamicProperty<>("docviewer.xls.max-size", DataSize.fromMegaBytes(30).toBytes());

    @Autowired
    private ConvertToHtmlHelper convertToHtmlHelper;

    private DocumentBuilderFactory documentBuilderFactory;

    @PostConstruct
    public void afterPropertiesSet() {
        try {
            documentBuilderFactory = DocumentBuilderFactory.newInstance();
            documentBuilderFactory.setNamespaceAware(true);
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @Override
    public ConvertResultInfo doConvert(InputStreamSource input, String contentType,
            TargetType convertTargetType, OutputStreamSource result,
            Option<String> password)
    {
        input.getFileO().filter(f -> f.length() > xlsMaxSize.get()).ifPresent(file -> {
            throw FileTooBigUserException.builder()
                    .maxLength(xlsMaxSize.get())
                    .actualLength(Option.of(file.length()))
                    .mimeType(MimeTypes.MIME_MICROSOFT_EXCEL)
                    .build();
        });
        try {
            switch (convertTargetType) {
                case HTML_ONLY:
                case HTML_WITH_IMAGES:
                case HTML_WITH_IMAGES_FOR_MOBILE: {
                    return doConvertToHtml(input, result, password);
                }
                case PLAIN_TEXT: {
                    return doConvertToText(input, result, password);
                }
                default: {
                    throw new UnsupportedOperationException("NYI: " + convertTargetType);
                }
            }
        } catch (EncryptedDocumentException exc) {
            throw PasswordProtectedException.atConvert(exc);
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    private ConvertResultInfo doConvertToHtml(
            InputStreamSource source, OutputStreamSource result, Option<String> password)
            throws Exception
    {
        final Document doc = documentBuilderFactory.newDocumentBuilder().newDocument();
        DocumentProperties properties = doConvertToHtml_w3cDom(source, doc, password);

        org.dom4j.Document dom4jDoc = XmlUtils2.convertToDom4j(doc);
        int pages = convertToHtmlHelper.splitAndPack(dom4jDoc, result, false, Option.empty(), false);

        return ConvertResultInfo.builder()
                .type(ConvertResultType.ZIPPED_HTML).pages(pages).properties(properties)
                .build();
    }

    private DocumentProperties doConvertToHtml_w3cDom(InputStreamSource source, Document doc, Option<String> password) {
        HSSFWorkbook excelWorkbook = XlsUtils.loadWorkbook(source, password);

        if (!XlsUtils.isSupported(excelWorkbook))
            throw new UnsupportedOperationException(
                    "Can't convert Excel workbooks with images or OLE objects");

        final HtmlDocumentFacade htmlDocumentFacade = new HtmlDocumentExtCssFacade(doc, Cf.map("c", "td."));

        ExcelToHtmlConverter excelToHtmlConverter = new ExcelToHtmlConverter(htmlDocumentFacade) {

            @Override
            protected boolean processCell(HSSFCell cell, Element tableCellElement,
                    int normalWidthPx, int maxSpannedWidthPx, float normalHeightPt)
            {
                ThreadLocalTimeout.check();

                boolean res = super.processCell(cell, tableCellElement, normalWidthPx, maxSpannedWidthPx,
                        normalHeightPt);

                // https://jira.yandex-team.ru/browse/DOCVIEWER-847
                if (tableCellElement.getElementsByTagName("div").getLength() > 0) {
                    // here is some copy/paste from super.processCell()
                    StringBuilder cellSizeStyle = new StringBuilder();

                    cellSizeStyle.append("min-width:");
                    cellSizeStyle.append(normalWidthPx);
                    cellSizeStyle.append("px;");
                    if (maxSpannedWidthPx != Integer.MAX_VALUE) {
                        cellSizeStyle.append("max-width:");
                        cellSizeStyle.append(maxSpannedWidthPx);
                        cellSizeStyle.append("px;");
                    }
                    cellSizeStyle.append("max-height:");
                    cellSizeStyle.append(normalHeightPt);

                    htmlDocumentFacade.addStyleClass(tableCellElement, getCssClassPrefixCell(),
                            cellSizeStyle.toString());
                }

                return res;
            }

            @Override
            protected int processRow(CellRangeAddress[][] mergedRanges, HSSFRow row,
                    Element tableRowElement)
            {
                ThreadLocalTimeout.check();
                return super.processRow(mergedRanges, row, tableRowElement);
            }

            @Override
            protected void processSheet(HSSFSheet sheet) {
                ThreadLocalTimeout.check();
                super.processSheet(sheet);
            }
        };
        excelToHtmlConverter.setCssClassPrefixCell("c");
        excelToHtmlConverter.setUseDivsToSpan(true);
        excelToHtmlConverter.setOutputColumnHeaders(false);
        excelToHtmlConverter.setOutputRowNumbers(false);
        excelToHtmlConverter.processWorkbook(excelWorkbook);

        return OleUtils.getDocumentProperties(excelWorkbook.getSummaryInformation());
    }

    private DocumentProperties doConvertToPdf_w3cDom(InputStreamSource source, Document doc, Option<String> password) {
        HSSFWorkbook excelWorkbook = XlsUtils.loadWorkbook(source, password);

        if (!excelWorkbook.getAllPictures().isEmpty()
                || !excelWorkbook.getAllEmbeddedObjects().isEmpty())
            throw new UnsupportedOperationException(
                    "Can't convert Excel workbooks with images or OLE objects");

        final ExtendedExcelToFoConverter excelToFoConverter = new ExtendedExcelToFoConverter(doc) {
            @Override
            protected boolean processCell(HSSFWorkbook workbook, HSSFCell cell,
                    Element tableCellElement, int normalWidthPx, int maxSpannedWidthPx,
                    float normalHeightPt)
            {
                ThreadLocalTimeout.check();
                return super.processCell(workbook, cell, tableCellElement, normalWidthPx,
                        maxSpannedWidthPx, normalHeightPt);
            }

            @Override
            protected int processRow(HSSFWorkbook workbook, CellRangeAddress[][] mergedRanges,
                    HSSFRow row, Element tableRowElement)
            {
                ThreadLocalTimeout.check();
                return super.processRow(workbook, mergedRanges, row, tableRowElement);
            }

            @Override
            protected float processSheet(HSSFWorkbook workbook, HSSFSheet sheet, Element flow) {
                ThreadLocalTimeout.check();
                return super.processSheet(workbook, sheet, flow);
            }
        };
        excelToFoConverter.setFontReplacer(new ExtendedFontReplacer());
        excelToFoConverter.setOutputColumnHeaders(false);
        excelToFoConverter.setOutputRowNumbers(false);
        excelToFoConverter.processWorkbook(excelWorkbook);

        return OleUtils.getDocumentProperties(excelWorkbook.getSummaryInformation());
    }

    private ConvertResultInfo doConvertToText(
            InputStreamSource source, OutputStreamSource result, Option<String> password)
    {
        ExcelExtractor excelExtractor = new ExcelExtractor(XlsUtils.loadWorkbook(source, password));
        excelExtractor.setFormulasNotResults(false);
        excelExtractor.setIncludeSheetNames(true);
        String text = excelExtractor.getText();
        result.write(text);
        return ConvertResultInfo.builder().type(ConvertResultType.PLAIN_TEXT).build();
    }

    @Override
    public boolean isSupported(TargetType convertTargetType) {
        return convertTargetType.isHtml()
                || convertTargetType == TargetType.PLAIN_TEXT;
    }
}
