package ru.yandex.chemodan.app.docviewer.copy;

import java.io.IOException;

import javax.annotation.PreDestroy;

import lombok.Data;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * 'https://sba.yandex.net/safety?client=saved-copy&url=https://www.muslim-library.com/dl/books/ru1162.pdf'
 */
public class BlockedUrlChecker implements AutoCloseable {

    private final static Logger logger = LoggerFactory.getLogger(BlockedUrlChecker.class);

    private final DynamicProperty<ListF<String>> blockedUrls = new DynamicProperty<>(
            "docviewer.url.serp.blocked", Cf.list()
    );

    private final DynamicProperty<Boolean> enabledSba = new DynamicProperty<>(
            "docviewer.url.serp.blocked.sba", true
    );

    private final BenderJsonParser<SbaResponse> responseParser = Bender.parser(SbaResponse.class);

    private final String sbaUrl;
    private final String sbaClent;
    private final HttpClient httpClient;

    public BlockedUrlChecker(String sbaUrl, String sbaClent, HttpClient httpClient) {
        this.sbaUrl = sbaUrl;
        this.sbaClent = sbaClent;
        this.httpClient = httpClient;
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    public boolean isBlocked(String url) {
        try {
            Option<String> blocked = blockedUrls.get().filter(url::contains).firstO();
            if (blocked.isPresent()) {
                return true;
            }
            if (StringUtils.isEmpty(sbaUrl) || !enabledSba.get()) {
                return false;
            }
            HttpUriRequest get = new HttpGet(UrlUtils.addParameter(sbaUrl, "client", sbaClent, "url", url));
            return ApacheHttpClientUtils.execute(get, httpClient, new Abstract200ExtendedResponseHandler<Boolean>() {

                @Override
                protected Boolean handle200Response(HttpResponse response) throws IOException {
                    SbaResponse sbaResponse = responseParser
                            .parseJson(InputStreamSourceUtils.wrap(response.getEntity().getContent()));
                    return sbaResponse.ban.getOrElse(false);
                }
            });
        } catch (Exception e) {
            logger.error("Can't check url " + url, e);
            return false;
        }
    }

    @Bendable
    @Data
    public static final class SbaResponse {
        @BenderPart(name = "ban_saved_copy", strictName = true)
        Option<Boolean> ban;
        @BenderPart(name = "url", strictName = true)
        Option<String> url;
    }

}
