package ru.yandex.chemodan.app.docviewer.copy;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.concurrent.CompletableFuture;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.utils.DateUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.archives.ArchiveContextFactory;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.copy.downloader.FileToDownloadViaMulcaClient;
import ru.yandex.chemodan.app.docviewer.copy.downloader.SynchronizedFileDownloader;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.PracticumApiManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.SchoolbookApiManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.StorageResourceInfoManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.TrackerApiManager;
import ru.yandex.chemodan.app.docviewer.copy.resourcemanagers.UslugiApiManager;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionDao;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey.SessionCopyPassword;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.FileTooBigUserException;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeChecker;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.Digester;
import ru.yandex.chemodan.app.docviewer.utils.DownloadRateInstrumentation;
import ru.yandex.chemodan.app.docviewer.utils.FileCopy;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.ExternalHttpClient;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.InternalHttpClient;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.MpfsHttpClient;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.app.docviewer.web.client.DocviewerForwardClient;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.commune.archive.ArchiveContext;
import ru.yandex.commune.archive.ArchiveEntry;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.inside.passport.tvm2.UserTicketHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileNotFoundIoException;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author vlsergey
 * @author akirakozov
 */
public class Copier {
    private static final Logger logger = LoggerFactory.getLogger(Copier.class);

    @Autowired
    private InternalHttpClient internalHttpClient;
    @Autowired(required = false)
    private ExternalHttpClient externalHttpClient;
    @Autowired
    private MpfsHttpClient mpfsHttpClient;
    @Autowired
    private Scheduler copierScheduler;
    @Autowired
    private MaxFileSizeChecker maxFileSizeChecker;
    @Autowired
    private MimeDetector mimeDetector;
    @Autowired
    private ArchiveManager archiveManager;
    @Autowired
    private ArchiveContextFactory archiveContextFactory;
    @Autowired
    private SynchronizedFileDownloader<FileToDownloadViaMulcaClient> fileDownloader;
    // XXX Another circular dependency!
    @Autowired
    private StateMachine stateMachine;
    @Autowired
    private Digester digester;
    @Autowired
    private SessionDao sessionDao;
    @Autowired
    @Qualifier("storedUriDao")
    private StoredUriDao storedUriDao;
    @Autowired
    private StorageResourceInfoManager copierResponseGetter;
    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private DocviewerForwardClient docviewerForwardClient;
    @Autowired
    private TrackerApiManager trackerApiManager;
    @Autowired
    private UslugiApiManager uslugiApiManager;
    @Autowired
    private SchoolbookApiManager schoolbookApiManager;
    @Autowired
    private PracticumApiManager practicumApiManager;

    @Value("${copier.archive.max.extract.depth}")
    private int maxExtractFromArchiveDepth;
    @Value("${mpfs.host}")
    private String mpfsHost;
    @Value("${mulca.gate.host}")
    private String mulcaGateHost;
    @Value("${wmi.host}")
    private String wmiHost;
    @Value("${mds.host-port}")
    private HostPort mdsHostPort;
    @Value("${tracker_api.host}")
    private String trackerApiHost;
    @Value("${schoolbook_api.host}")
    private String schoolbookApiHost;
    @Value("${practicum_api.host}")
    private String practicumApiHost;

    @Value("${copier.head.timeout}")
    private Duration headTimeout;

    private CompletableFuture<Boolean> needToDownloadFileInternal(final URI uri, final Instant ifModifiedSince,
            final boolean isExternalUri)
    {
        String host = UriUtils.getHost(uri);
        boolean isMpfsUrl = mpfsHost.equals(host);
        if (isMpfsUrl || isExternalUri) {
            if (isVersioning(uri)) {
                logger.info("Skip download {} cause it has version", uri);
                return CompletableFuture.completedFuture(false);
            }
            HttpHead httpHead = new HttpHead(uri);
            httpHead.addHeader(HttpHeaders.IF_MODIFIED_SINCE, DateUtils.formatDate(ifModifiedSince.toDate()));
            final int headTimeoutMillis = Math.toIntExact(headTimeout.getMillis());
            httpHead.setConfig(RequestConfig.custom()
                    .setRedirectsEnabled(false)
                    .setConnectTimeout(headTimeoutMillis)
                    .setSocketTimeout(headTimeoutMillis)
                    .setConnectionRequestTimeout(headTimeoutMillis)
                    .build());
            HttpClient httpClient = isMpfsUrl ? mpfsHttpClient : externalHttpClient;

            return CompletableFuture.supplyAsync(YandexCloudRequestIdHolder.supplyWithYcrid(() -> {
                final Instant startTime = TimeUtils.now();
                try {
                    return ApacheHttpClientUtils.execute(httpHead,
                            httpClient,
                            response -> {
                                int statusCode = response.getStatusLine().getStatusCode();
                                LoggerEventsRecorder.saveCheckSourceFileEvent(
                                        uri, response.getStatusLine().toString(), TimeUtils.toDurationToNow(startTime));
                                switch (statusCode) {
                                    case HttpStatus.SC_NOT_FOUND:
                                        throw new UserException(ErrorCode.FILE_NOT_FOUND);
                                    case HttpStatus.SC_FORBIDDEN:
                                        throw new UserException(ErrorCode.FILE_IS_FORBIDDEN);
                                    default:
                                        return statusCode != HttpStatus.SC_NOT_MODIFIED;
                                }
                            }
                    );
                } catch (RuntimeIoException e) {
                    logger.info("The target server failed to respond to HEAD request", e);
                    return true; //falling back to decide to download file
                }
            }));
        } else if (mulcaGateHost.equals(host) || wmiHost.equals(host) ||
                mdsHostPort.getHost().getIdn().getNormalized().equals(host))
        {
            return CompletableFuture.completedFuture(false);
        } else if (trackerApiHost.equals(host)) {
            return CompletableFuture.completedFuture(!trackerApiManager.isUriAccessible(uri));
        } else if (uslugiApiManager.isUrlMatched(uri.toString())) {
            return CompletableFuture.completedFuture(!uslugiApiManager.isUriAccessible(uri));
        } else if (schoolbookApiHost.equals(host)) {
            return CompletableFuture.completedFuture(!schoolbookApiManager.isUriAccessible(uri));
        } else if (practicumApiHost.equals(host)) {
            return CompletableFuture.completedFuture(!practicumApiManager.isUriAccessible(uri));
        } else {
            logger.debug("About to download: {}", uri);
            return CompletableFuture.completedFuture(true);
        }
    }

    private boolean isVersioning(java.net.URI uri) {
        return UrlUtils.getQueryParameterFromUrl(uri.toString(), "version_id").isPresent();
    }

    public CompletableFuture<Boolean> needToDownloadFile(ActualUri uri, Instant ifModifiedSince,
            boolean isExternalUri)
    {
        if (isEnableNativeUrlFetching()) {
            return CompletableFuture.completedFuture(false);
        } else {
            return needToDownloadFileInternal(uri.getUri(), ifModifiedSince, isExternalUri);
        }
    }

    private TempFileInfo downloadAndSaveInternal(URI uri, boolean isExternalUri, MaxFileSizeChecker sizeChecker) {
        Validate.isTrue(UriUtils.getHostO(uri).isPresent(), "Empty host, url: ", uri);

        Option<StorageResourceInfo> copierResponse = copierResponseGetter.getStorageResourceInfoResponse(uri);
        TempFileInfo result;

        if (dowloadViaMulcagateOrUnistorage(copierResponse)) {
            result = downloadAndSaveFromStorage(uri, copierResponse.get(), sizeChecker);
            DownloadRateInstrumentation.downloadRateMetric.inc(result.length(), "raw");
        } else if (copierResponse.exists(info -> info.getId().isRawUrl())) {
            result = ApacheHttpClientUtils.execute(new HttpGet(copierResponse.get().getId().getRawUrl()),
                    isExternalUri ? externalHttpClient : internalHttpClient,
                    new CopierResponseHandler(uri, Option.empty(), sizeChecker, Option.empty(), mimeDetector));
            DownloadRateInstrumentation.downloadRateMetric.inc(result.length(), "raw-not-mulca");
        } else {
            HttpGet get = prepareHttpRequest(uri);
            result = ApacheHttpClientUtils.execute(get,
                    isExternalUri ? externalHttpClient : internalHttpClient,
                    new CopierResponseHandler(uri, Option.empty(), sizeChecker, Option.empty(), mimeDetector));
            DownloadRateInstrumentation.downloadRateMetric.inc(result.length(), "raw-not-mulca");
        }
        return result;
    }

    private boolean dowloadViaMulcagateOrUnistorage(Option<StorageResourceInfo> copierResponse) {
        return copierResponse.isMatch(response -> response.getStorageId().isMulcaId());
    }

    private HttpGet prepareHttpRequest(URI uri) {
        HttpGet get = new HttpGet(uri);
        uriHelper.findProviderByActualUri(new ActualUri(uri))
                .ifPresent(provider -> provider.getSpecificHeaders().forEach(get::setHeader));
        return get;
    }

    private TempFileInfo downloadAndSaveFromStorage(
            URI uri, StorageResourceInfo copierResponse, MaxFileSizeChecker sizeChecker)
    {
        return fileDownloader.download(uri, copierResponse, sizeChecker);
    }

    private TempFileInfo downloadAndSaveByNativeUrl(ActualUri uri) throws IOException {
        Validate.isTrue(EnvironmentType.getActive() == EnvironmentType.TESTS
                || EnvironmentType.getActive() == EnvironmentType.DEVELOPMENT);
        try {
            Option<String> contentDispositionFilename = Option.empty();

            File2 tempFile = FileUtils.createEmptyTempFile("downloaded", ".tmp");
            try (
                    InputStream in = new UrlInputStreamSource(UriUtils.toUrl(uri.getUri())).getInput();
                    OutputStream out = new FileOutputStreamSource(tempFile).getOutput()
            )
            {
                IoUtils.copy(in, out);
            }

            return new TempFileInfo(Option.of(mimeDetector.getMimeType(tempFile)), contentDispositionFilename,
                    new FileCopy(tempFile), uri.getUri(), true, Option.empty());
        } catch (FileNotFoundIoException | FileNotFoundException exc) {
            throw new UserException(ErrorCode.FILE_NOT_FOUND);
        }
    }

    private TempFileInfo downloadAndSave(
            ActualUri uri, boolean isExternalUri, boolean warmUp, MaxFileSizeChecker sizeChecker)
    {
        ActualUri uriNoPath = uri.withoutArchivePath();

        final Instant startTime = TimeUtils.now();
        TempFileInfo res = null;
        try {
            res = isEnableNativeUrlFetching() ?
                    downloadAndSaveByNativeUrl(uriNoPath) :
                    downloadAndSaveInternal(uriNoPath.getUri(), isExternalUri, sizeChecker);

            LoggerEventsRecorder.saveCopyEvent(res.getUriForLogging(), res.getReportedContentType(),
                    res.getContentDispositionFilename().map(File2::getExtensionFromPath),
                    res.getLocalCopy().getFile().length(), TimeUtils.toDurationToNow(startTime),
                    res.getCacheResult(), warmUp);

            return res;
        } catch (Exception e) {
            if (res != null) {
                res.getLocalCopy().deleteFileIfPossible();
            }
            throw ExceptionUtils.translate(e);
        }
    }

    private TempFileInfo extractFileByPartialPath(String partialPath, TempFileInfo info, String pathPrefix,
            Option<String> password, MaxFileSizeChecker sizeChecker)
    {
        final Instant startTime = TimeUtils.now();
        File2 extracted = null;
        try {
            String mimeTypeByFilename = mimeDetector.getMimeTypeByFilename(partialPath);
            ArchiveContext context = archiveContextFactory.fileExtractingContext(password, pathPrefix);

            ArchiveListing listing = archiveManager.listArchive(info.getLocalCopy().getFile(), context);
            ArchiveEntry entry = listing.getEntries().find(ArchiveEntry.pathEqualsToF(partialPath))
                    .getOrThrow(
                            () -> new UserException(ErrorCode.FILE_NOT_FOUND, "Not found in archive: " + partialPath));

            entry.getSize().ifPresent(size -> sizeChecker.check(size, mimeTypeByFilename));

            // XXX this extracted temp file is now placed to generic tmp directory, not docviewer's one
            try {
                File2 localFile = info.getLocalCopy().getFile();
                logger.info("About to extract file from {} by path: {}", localFile, partialPath);
                extracted = archiveManager.extractOneToTempFile(localFile, context, partialPath);

                // double check -- XXX archive manager now accepts per-extraction context -- remove this
                sizeChecker.check(extracted.length(), mimeTypeByFilename);
            } finally {
                info.getLocalCopy().deleteFileIfPossible();
            }


            String readablePath = entry.getReadablePath();
            String filenameInArchive = readablePath.isEmpty()
                    ? "archive_contents"
                    : (readablePath.contains("/") //  fallback for archives w/o listing
                    ? StringUtils.substringAfterLast(readablePath, "/")
                    : readablePath);

            info = new TempFileInfo(Option.of(mimeDetector.getMimeType(extracted)),
                    StringUtils.notEmptyO(filenameInArchive), new FileCopy(extracted), info.getUriForLogging(), false,
                    Option.empty());

            LoggerEventsRecorder
                    .saveExtractEvent(TimeUtils.toDurationToNow(startTime), partialPath, extracted.length());

            return info;
        } catch (Exception e) {
            LoggerEventsRecorder.saveExtractFailedEvent(e, TimeUtils.toDurationToNow(startTime), partialPath);
            if (extracted != null) {
                extracted.deleteRecursiveQuietly();
            }
            if (e instanceof UserException) {
                throw (UserException) e;
            } else {
                throw new UserException(ErrorCode.ARCHIVE_EXTRACTION_ERROR, e);
            }
        }
    }




    private TempFileInfo extractFileByFullPath(ActualUri uri, String archivePath, TempFileInfo info,
            Option<String> sessionId, MaxFileSizeChecker sizeChecker)
    {
        logger.debug("Extracting file from archive using full path '{}'...", archivePath);

        int depth = 0;
        String pathPrefix = "";
        while (StringUtils.isNotEmpty(archivePath)) {
            Validate.isTrue(archivePath.startsWith("//"), "Wrong archive path: ", archivePath);

            if (depth < maxExtractFromArchiveDepth) {
                String nestedLookup = archivePath.substring(2);
                String partialPath = StringUtils.substringBefore(nestedLookup, "//");
                String pathPrefixFinal = pathPrefix;

                Option<String> password = sessionId.map(s ->
                        sessionDao.findValidValue(s, new SessionCopyPassword(uri.withArchivePath(pathPrefixFinal)))
                ).getOrElse(Option.empty());

                info = extractFileByPartialPath(partialPath, info, pathPrefix, password, sizeChecker);

                if (password.isPresent()) {
                    storedUriDao.updatePasswordRaw(uri, pathPrefix, password.get());
                }

                archivePath = nestedLookup.substring(partialPath.length());
                pathPrefix += "//" + partialPath;
            } else {
                // TODO: create separate error for this case?
                info.getLocalCopy().deleteFileIfPossible();
                throw new UserException(ErrorCode.ARCHIVE_EXTRACTION_ERROR);
            }
            depth++;
        }

        return info;
    }

    public TempFileInfo downloadAndSaveAndExtract(ActualUri uri, boolean isExternalUri, Option<String> sessionId,
            boolean warmUp, MaxFileSizeChecker sizeChecker)
    {
        Option<String> archivePath = uri.getArchivePathO();
        TempFileInfo res = downloadAndSave(uri, isExternalUri, warmUp, sizeChecker);

        if (archivePath.isPresent()) {
            res = extractFileByFullPath(uri, archivePath.get(), res, sessionId, sizeChecker);
        }

        return res;
    }

    public TempFileInfo downloadAndSaveAndExtract(
            ActualUri uri, boolean isExternalUri, Option<String> sessionId, boolean warmUp)
    {
        return downloadAndSaveAndExtract(uri, isExternalUri, sessionId, warmUp, maxFileSizeChecker);
    }


    private void copyImpl(CopyInfo info) {
        boolean isFileUsed = false;
        Option<TempFileInfo> tempFileInfoO = Option.empty();
        Option<String> resultContentType = info.getContentType();
        try {
            logger.debug("Copy resource '{}' to temporary place and calculate hashes... ", info);
            stateMachine.onCopyBegin(info.uri);

            tempFileInfoO = Option.of(downloadAndSaveAndExtract(info.getUri(), info.isExternalUri(),
                    Option.of(info.getSessionId()), info.getSource().isWarmUp()));
            Option<String> reportedContentType = tempFileInfoO.get().getReportedContentType();

            resultContentType = reportedContentType.isPresent() ? reportedContentType : resultContentType;

            String fileId = digester.calculateDigestId(tempFileInfoO.get().getLocalCopy().getFile());

            CopiedFileInfo copiedInfo = new CopiedFileInfo(info.uri, resultContentType,
                    tempFileInfoO.get().getContentDispositionFilename(), tempFileInfoO.get().getLocalCopy(),
                    fileId, tempFileInfoO.get().isUseUriForMimeType(), tempFileInfoO.get().getSerpLastAccess(),
                    info.getStartInfo(), info.createRestoreUri());
            isFileUsed = stateMachine.onCopyDone(copiedInfo, info.getSessionId());
        } catch (FileTooBigUserException e) {
            try {
                logger.warn("About to forward request {}", info, e);
                if (!callToRemote(info, resultContentType)) {
                    handleCopyError(e, info);
                }
            } catch (Exception in) {
                logger.warn("Error during forwarding request {}", info, in);
                LoggerEventsRecorder.saveForwardToStartInternalFailedEvent(
                        info.getSource().getOriginalUrl(), info.getSource().getUid(),
                        TimeUtils.toDurationToNow(info.getStartInfo().getStartTime()), in);
                stateMachine.onCopyError(info.uri, in);
            }
            isFileUsed = false;
        } catch (Exception e) {
            handleCopyError(e, info);
            isFileUsed = false;
        } finally {
            if (!isFileUsed) {
                tempFileInfoO.map(TempFileInfo::getLocalCopy).ifPresent(FileCopy::deleteFileIfPossible);
            }
        }
    }

    private void handleCopyError(Exception e, CopyInfo info) {
        ActualUri uriNoPath = info.getUri().withoutArchivePath();
        Option<Long> lengthO = e instanceof FileTooBigUserException ?
                ((FileTooBigUserException) e).getActualLength() : Option.empty();
        LoggerEventsRecorder.saveCopyFailedEvent(
                uriNoPath, e, TimeUtils.toDurationToNow(info.getStartInfo().getStartTime()),
                lengthO,
                info.getSource().isWarmUp());
        stateMachine.onCopyError(info.uri, e);
    }

    private boolean callToRemote(CopyInfo info, Option<String> resultContentType) {
        if (info.isForwarded()) {
            return false;
        }
        docviewerForwardClient.forwardToStartInternal(info, resultContentType);
        LoggerEventsRecorder.saveForwardToStartInternalEvent(
                info.getSource().getOriginalUrl(), info.getSource().getUid(),
                TimeUtils.toDurationToNow(info.getStartInfo().getStartTime()), "ok");
        return true;
    }

    public void scheduleCopy(CopyInfo info) {
        copierScheduler.scheduleGlobalTask(info.getUri().getUriString(), () -> {
            UserTicketHolder.withUserTicketO(info.tvmUserTicket, () -> copyImpl(info));
            return null;
        }, 1);
    }

    public boolean isEnableNativeUrlFetching() {
        return internalHttpClient.isEnableNativeUrlFetching();
    }

    // for testing purpose
    public void setEnableNativeUrlFetching(boolean enableNativeUrlFetching) {
        internalHttpClient.setEnableNativeUrlFetching(enableNativeUrlFetching);
    }
}
