package ru.yandex.chemodan.app.docviewer.copy;

import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;

import javax.mail.internet.ContentType;

import lombok.Data;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeChecker;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.FileCopy;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.HttpUtils2;
import ru.yandex.chemodan.app.docviewer.utils.MimeUtils;
import ru.yandex.inside.mulca.MulcaUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
@Data
public class CopierResponseHandler implements ResponseHandler<TempFileInfo> {
    private static final Logger logger = LoggerFactory.getLogger(CopierResponseHandler.class);

    private final URI uri;
    private final Option<String> specifiedContentType;
    private final MaxFileSizeChecker maxFileSizeChecker;
    private final Option<String> encoding;
    private final MimeDetector mimeDetector;

    @Override
    public TempFileInfo handleResponse(HttpResponse response) {
        checkHttpStatusCode(response);

        final HttpEntity entity = response.getEntity();
        final long contentLength = entity.getContentLength();
        final Option<String> downloadedContentType = specifiedContentType.orElse(findContentType(entity));

        String contentType = downloadedContentType.getOrElse("");

        checkMaxFileSize(contentLength, contentType);

        Option<String> contentDispositionFilename = detectContentType(response, entity);
        final File2 downloaded = FileUtils.createEmptyTempFile("downloaded", ".tmp");
        try {
            InputStream in = encoding.isPresent() ?
                    MulcaUtils.decode(entity.getContent(), encoding.get()) :
                    entity.getContent();
            FileUtils.copyAndCloseQuitely(new InputStreamX(in).limit(maxFileSizeChecker.getMaxFileLength(contentType).toBytes() + 1),
                    new FileOutputStream(downloaded.getFile()));
            checkFileSize(downloaded.length(), contentType);
            logger.debug("Downloaded file {}, size: {}", downloaded, downloaded.length());
        } catch (Exception e) {
            downloaded.deleteRecursiveQuietly();
            throw ExceptionUtils.translate(e);
        }

        Option<Instant> serpLastAccess = getSerpLastAccessHeader(response);
        return new TempFileInfo(Option.of(mimeDetector.getMimeType(downloaded)), contentDispositionFilename,
                new FileCopy(downloaded), uri, true, serpLastAccess);
    }

    private void checkHttpStatusCode(HttpResponse response) {
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_404_NOT_FOUND) {
            throw new UserException(ErrorCode.FILE_NOT_FOUND, getMessage(response));
        }
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_403_FORBIDDEN) {
            throw new UserException(ErrorCode.FILE_IS_FORBIDDEN, getMessage(response));
        }
        if (!HttpStatus.is2xx(response.getStatusLine().getStatusCode())) {
            String msg = "Unable to retrieve file, remote server response was " + getMessage(response);
            logger.info(msg);
            throw new UserException(ErrorCode.UNABLE_TO_RETRIEVE_FILE, msg);
        }
    }

    protected String getMessage(HttpResponse response) {
        StringBuilder message = new StringBuilder(response.getStatusLine().toString());
        if (response.getEntity() != null) {
            try (Reader reader = new InputStreamReader(response.getEntity().getContent());
                 BufferedReader br = new BufferedReader(reader)) {
                message.append("; response= ");
                br.lines().filter(StringUtils::isNotEmpty).forEach(message::append);
            } catch (Exception e) {
                //ignore here
            }
        }
        return message.toString();
    }

    private Option<String> detectContentType(HttpResponse response, HttpEntity entity) {
        Option<String> contentDispositionFilename = findContentDisposition(response);

        if (contentDispositionFilename.isPresent()) {
            return contentDispositionFilename;
        }

        final Header contentTypeHeader = entity.getContentType();
        if (contentTypeHeader != null && StringUtils.isNotEmpty(contentTypeHeader.getValue())) {
            try {
                ContentType contentType = new ContentType(contentTypeHeader.getValue());
                String fileName = contentType.getParameter("filename");
                contentDispositionFilename = StringUtils.notBlankO(fileName);
            } catch (Exception exc) {
                logger.debug("Unable to extract filename from content type header from '"
                        + uri + "': " + exc, exc);
            }
        }

        return contentDispositionFilename;
    }

    private void checkFileSize(long contentLength, String contentType) {

        if (contentLength < 1) {
            throw new UserException(ErrorCode.FILE_IS_EMPTY);
        } else {
            checkMaxFileSize(contentLength, contentType);
        }
    }

    private void checkMaxFileSize(long contentLength, String contentType) {
        long estimatedFileLength = encoding
                .map(enc -> MimeUtils.getEstimatedFileLength(contentLength, enc))
                .getOrElse(contentLength);

        maxFileSizeChecker.check(estimatedFileLength, contentType);
    }

    private Option<String> findContentDisposition(HttpResponse response) {
        Option<String> contentDispositionFilename = Option.empty();
        try {
            Header contentDispositionHeader = response.getFirstHeader(HttpUtils2.HEADER_CONTENT_DISPOSITION);
            if (contentDispositionHeader != null) {
                contentDispositionFilename = HttpUtils2.getFilenameFromContentDisposition(
                        contentDispositionHeader.getValue());
            }
        } catch (Exception exc) {
            logger.error("Unable to retrieve content disposition from response: " + exc, exc);
        }

        return contentDispositionFilename;
    }

    private Option<String> findContentType(final HttpEntity entity) {
        Option<String> downloadedContentType = Option.empty();
        final Header contentTypeHeader = entity.getContentType();
        if (contentTypeHeader != null) {
            downloadedContentType = HttpUtils2.toContentType(uri.toString(),
                    contentTypeHeader.getValue());
        }

        return downloadedContentType;
    }

    private Option<Instant> getSerpLastAccessHeader(HttpResponse response) {
        try {
            return Option.ofNullable(response.getFirstHeader(HttpUtils2.HEADER_LAST_ACCESS))
                    .map(Header::getValue)
                    .map(Cf.Long.parseF())
                    .map(sec -> new Instant(sec * 1000));
        } catch (Exception e) {
            logger.warn("Couldn't parse LastAccess header: " + e.getMessage());
        }
        return Option.empty();
    }
}
