package ru.yandex.chemodan.app.docviewer.copy;

import java.util.Base64;

import com.google.common.primitives.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.DigestUtils;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.io.http.UrlUtils;

/**
 * @author bursy
 */
public class SerpUrlChecker {
    private static final Logger logger = LoggerFactory.getLogger(SerpUrlChecker.class);

    private static final String SIGN_PARAMETER = "sign";
    private static final String KEYNO_PARAMETER = "keyno";
    private static final String TM = "tm";

//    public final DynamicProperty<Integer> serpTtl =
//            new DynamicProperty<>("docviewer.url.serp.ttl.seconds", 5 * 24 * 60 * 60);

    private static final int SIGN_LENGTH = 32;

    private final String encodedSerpKey;

    public SerpUrlChecker(String encodedSerpKey) {
        this.encodedSerpKey = encodedSerpKey;
    }

    /**
     * Verifies serp link to docviewer using serp signatures in given query.
     * Expects the parameters to be passed from serp verbatim.
     * @param paramsString whole parameter string from serp
     * @return whether the signatures are correct
     */
    public boolean verify(String paramsString) {
        Tuple2List<String, String> params = UrlUtils.urlParametersToTuple2ListNoUrlDecode(paramsString);
        MapF<String, String> paramsMap = params.toMap();

        Option<String> signO = paramsMap.getO(SIGN_PARAMETER);
        Option<String> keynoO = paramsMap.getO(KEYNO_PARAMETER);
        Option<String> tm = paramsMap.getO(TM);

        if (!signO.isPresent() || !keynoO.isPresent()) {
            logger.error("Url cannot be verified: it doesn't have required parameters from serp {}", paramsString);
            return false;
        }

        if (signO.get().length() != SIGN_LENGTH) {
            logger.error("Passed signature has wrong size {}", paramsString);
            return false;
        }

//        disabled time checking
//        if (tm.isPresent() && !checkTmTtl(Long.valueOf(tm.get()))) {
//            logger.error("Outdated url {}", paramsString);
//            return false;
//        }

        String signBase = paramsString.substring(0, paramsString.indexOf("&" + SIGN_PARAMETER + "="));

        Option<SerpKey> keyO = retrieveKey(Integer.parseInt(keynoO.get()));
        if (!keyO.isPresent()) {
            logger.error("Unable to retrieve key by keyno: " + keynoO.get());
            return false;
        }

        byte[] bytes = Bytes.concat(signBase.getBytes(), keyO.get().bytes);
        String hash = DigestUtils.md5DigestAsHex(bytes);

        return hash.equals(signO.get());
    }

//    protected boolean checkTmTtl(Long ttl) {
//        return new Instant(Duration.standardSeconds(ttl).getMillis())
//                .plus(Duration.standardSeconds(serpTtl.get()))
//                .isAfter(Instant.now());
//    }

    private Option<SerpKey> retrieveKey(int keyno) {
        // TODO: need to support all keynos
        if (keyno == 0) {
            return Option.of(new SerpKey(encodedSerpKey));
        }

        return Option.empty();
    }

    private static class SerpKey {
        private final byte[] bytes;

        public SerpKey(String encodedKey) {
            this.bytes = Base64.getDecoder().decode(encodedKey);
        }
    }
}
