package ru.yandex.chemodan.app.docviewer.copy;

import lombok.AllArgsConstructor;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.params.ClientPNames;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.copy.provider.DocumentUrlProvider;
import ru.yandex.chemodan.app.docviewer.copy.provider.MulcaUrlProvider;
import ru.yandex.chemodan.app.docviewer.copy.provider.RealHttpUrlProvider;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.DotReplacingHostnameProcessor;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ForbiddenException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClient4Utils;
import ru.yandex.misc.io.http.apache.v4.InstrumentedDefaultHttpClient;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author akirakozov
 */
@AllArgsConstructor
public class UriHelper {

    private static final Logger logger = LoggerFactory.getLogger(UriHelper.class);

    private final ListF<DocumentUrlProvider> urlProviders;
    private final SerpUrlChecker serpUrlChecker;
    private boolean disableOriginalUrlCheck;
    private final Duration timeout;
    private final BlockedUrlChecker blockedUrlChecker;

    public void checkOriginalUrlAllowed(String originalUrl, PassportUidOrZero uid, Option<String> serpParams) {
        if (disableOriginalUrlCheck) return;

        switch (checkOriginalUrlAllowedInner(originalUrl, uid, serpParams)) {
            case BAD_REQUEST:
                throw new IllegalArgumentException("Specified URL '" + originalUrl + "' is incorrect");
            case FORBIDDEN:
                throw new ForbiddenException("DocViewer settings doesn't allow document retrieving " +
                        "from the URL '" + originalUrl + "'");
            case NOT_IMPLEMENTED:
                throw new UnsupportedOperationException("Not implemented");
            case OK:
                break;
            default:
                Check.fail();
        }
    }

    public boolean isExternalUrl(String url) {
        return isExternalUrl(url, this::findProviderByUrl);
    }

    public boolean isExternalUrl(ActualUri actualUri) {
        return isExternalUrl(actualUri, this::findProviderByActualUri);
    }

    private <T> boolean isExternalUrl(T url, Function<T, Option<DocumentUrlProvider>> findProviderF) {
        return findProviderF.apply(url).isMatch(RealHttpUrlProvider.class::isInstance);
    }

    private Option<DocumentUrlProvider> findProviderByUrl(String originalUrl) {
        return urlProviders.filter(DocumentUrlProvider.isSupportedUrlF(originalUrl)).firstO();
    }

    public Option<DocumentUrlProvider> findProviderByActualUri(ActualUri actualUri) {
        ListF<DocumentUrlProvider> providersWithDefaultRealHttpPriovider = urlProviders
                .filter(DocumentUrlProvider.isSupportedActualUriF(actualUri));

        ListF<DocumentUrlProvider> specificProviders = providersWithDefaultRealHttpPriovider
                .filter(provider -> !(provider instanceof RealHttpUrlProvider));

        return specificProviders.isNotEmpty() ? specificProviders.firstO() : providersWithDefaultRealHttpPriovider.firstO();
    }

    UriCheckResult checkOriginalUrlAllowedInner(String originalUrl, PassportUidOrZero uid, Option<String> serpParams) {
        logger.debug("Checking if uri '{}' is allowed...", originalUrl);

        String scheme = getScheme(originalUrl);
        if (StringUtils.isBlank(originalUrl)) {
            return UriCheckResult.BAD_REQUEST;
        } else if (StringUtils.isEmpty(scheme)) {
            logger.debug("Result is NOT_IMPLEMENTED because URL '{}' have empty scheme part", originalUrl);
            return UriCheckResult.NOT_IMPLEMENTED;
        } else if (isYaTeamServiceUrl(originalUrl) && !isYaTeamUser(uid)) {
            return UriCheckResult.FORBIDDEN;
        } else if (isAllowedUrl(originalUrl, serpParams)) {
            return UriCheckResult.OK;
        }

        return UriCheckResult.FORBIDDEN;
    }

    private String getScheme(String url) {
        return url.startsWith("file:") ? "file" : // for test resource urls
                url.contains("://") ? StringUtils.substringBefore(url, "://") : "";
    }

    private boolean isAllowedUrl(String originalUrl, Option<String> serpParams) {
        if (MulcaUrlProvider.isMulcaUrl(originalUrl)) {
            return false;
        } else {
            Option<DocumentUrlProvider> providerByUrl = findProviderByUrl(originalUrl);
            return providerByUrl.isPresent() &&
                    (!(providerByUrl.get() instanceof RealHttpUrlProvider) || isSafeUrl(originalUrl, serpParams));
        }
    }

    private boolean isSafeUrl(final String url, Option<String> serpParams) {
        if (blockedUrlChecker.isBlocked(url)) {
            return false;
        }
        if (!serpParams.isPresent()) {
            logger.debug("Serp params not passed, falling back to serp request");
            return checkUrlInSerp(url);
        }

        if (serpUrlChecker.verify(serpParams.get())) {
            return true;
        } else {
            logger.warn("Serp hash check failed for params {}, falling back to serp request", serpParams.get());
        }

        return checkUrlInSerp(url);
    }

    private boolean checkUrlInSerp(String url) {
        try {
            String yaSerpUrl = externalUrlToYaSerpUrl(url);

            HttpClient httpClient = ApacheHttpClient4Utils.singleConnectionClient(Timeout.valueOf(timeout));
            ((InstrumentedDefaultHttpClient)httpClient).setHostnameProcessor(new DotReplacingHostnameProcessor());
            HttpHead httpHead = new HttpHead(yaSerpUrl);
            httpHead.getParams().setBooleanParameter(ClientPNames.HANDLE_REDIRECTS, false);

            return ApacheHttpClient4Utils.execute(httpHead, httpClient,
                    response -> response.getStatusLine().getStatusCode() == HttpStatus.SC_OK,
                    yaSerpUrl);
        } catch (Exception e) {
            logger.error("Can't check url: " + url, e);
            return true;
        }

    }

    private String externalUrlToYaSerpUrl(String externalUrl) {
        String yaSerpUrl = externalUrl.replaceFirst("http", "ya-serp");
        return rewrite(DocumentSourceInfo.builder().originalUrl(yaSerpUrl).uid(PassportUidOrZero.zero()).build()).getUriString();
    }

    private boolean isYaTeamServiceUrl(String originalUrl) {
        return findProviderByUrl(originalUrl)
                .map(DocumentUrlProvider::isYaTeamServiceProvider)
                .getOrElse(false);
    }

    private boolean isYaTeamUser(PassportUidOrZero uid) {
        return uid.isAuthenticated() && uid.toPassportUid().isYandexTeamRu();
    }

    public ActualUri rewriteForTests(String url, PassportUidOrZero uid) {
        return rewrite(DocumentSourceInfo.builder().originalUrl(url).uid(uid).build());
    }

    public ActualUri rewrite(DocumentSourceInfo source) {
        Option<DocumentUrlProvider> urlProvider = findProviderByUrl(source.getOriginalUrl());
        String resultString = urlProvider.isPresent() ?
                urlProvider.get().rewriteUrl(source) : source.getOriginalUrl();

        // archive-path should be the last parameter of the actual-uri because it can be updated
        if (source.getArchivePath().isPresent()) {
            resultString = UrlUtils.addParameter(resultString, "archive-path", source.getArchivePath().get());
        }

        return new ActualUri(UriUtils.toUriSafe(resultString));
    }

    public Option<DiskResourceId> getDiskResourceId(DocumentSourceInfo source) {
        return getDiskResourceId(source, rewrite(source));
    }

    public Option<DiskResourceId> getDiskResourceId(DocumentSourceInfo source, ActualUri uri) {
        Option<DocumentUrlProvider> urlProvider = findProviderByUrl(source.getOriginalUrl());
        if (!urlProvider.isPresent()) {
            urlProvider = findProviderByActualUri(uri);
        }
        return urlProvider.isPresent() ? urlProvider.get().getDiskResourceId(uri) : Option.empty();
    }

    public Option<DiskResourceId> getDiskResourceId(ActualUri uri) {
        return findProviderByActualUri(uri).flatMapO(provider -> provider.getDiskResourceId(uri));
    }

    // for tests
    public void setDisableOriginalUrlCheck(boolean disableOriginalUrlCheck) {
        this.disableOriginalUrlCheck = disableOriginalUrlCheck;
    }
}
