package ru.yandex.chemodan.app.docviewer.copy.downloader;

import java.net.URI;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.copy.CacheResult;
import ru.yandex.chemodan.app.docviewer.copy.StorageResourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.TempFileInfo;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeChecker;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.utils.FileCopy;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class CachingFileDownloader<T extends FileToDownload> implements FileDownloader<T> {
    private static final Logger logger = LoggerFactory.getLogger(CachingFileDownloader.class);

    @Autowired
    private MimeDetector mimeDetector;

    @Autowired
    private TemporaryFileCache temporaryFileCache;

    private final FileDownloader<T> delegate;

    public CachingFileDownloader(FileDownloader<T> delegate) {
        this.delegate = delegate;
    }

    @Override
    public T buildFileToDownload(URI uri, StorageResourceInfo copierResponse) {
        return delegate.buildFileToDownload(uri, copierResponse);
    }

    @Override
    public TempFileInfo download(T fileToDownload, MaxFileSizeChecker sizeChecker) {
        FileLink fileLink = fileToDownload.toFileLink();
        Option<File2> cachedFile = temporaryFileCache.get(fileLink);
        boolean needCaching = isNeedCaching(fileToDownload);

        if (cachedFile.isPresent()) {
            return new TempFileInfo(Option.of(mimeDetector.getMimeType(cachedFile.get())),
                    fileToDownload.getContentDispositionFilename(), new FileCopy(cachedFile.get(), false),
                    fileToDownload.getUri(), false, CacheResult.get(needCaching, true), Option.empty());
        }

        TempFileInfo info = delegate.download(fileToDownload, sizeChecker)
                .withCacheResult(CacheResult.get(fileToDownload.isArchive(), false));

        if (!needCaching && MimeDetector.isArchive(info.getReportedContentType().getOrElse(""))) {
            logger.info("File is now treated an archive");
            // no need to store determined refinedMimeType as a chosenContentType
            needCaching = true;
        }

        if (needCaching) {
            File2 file = info.getLocalCopy().getFile();
            file = temporaryFileCache.putInCache(fileLink, file);
            info = info.withLocalCopy(new FileCopy(file, false));
        }

        return info;
    }

    boolean isNeedCaching(T fileToDownload) {
        return fileToDownload.isArchive();
    }

    boolean isCached(FileLink fileLink) {
        return temporaryFileCache.get(fileLink).isPresent();
    }
}

