package ru.yandex.chemodan.app.docviewer.copy.downloader;

import java.net.URI;

import org.apache.http.client.methods.HttpGet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.copy.CacheResult;
import ru.yandex.chemodan.app.docviewer.copy.CopierResponseHandler;
import ru.yandex.chemodan.app.docviewer.copy.StorageResourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.TempFileInfo;
import ru.yandex.chemodan.app.docviewer.copy.storage.StorageId;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeChecker;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.MulcaHttpClient;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.mulca.MulcaPartHeaders;
import ru.yandex.inside.mulca.MulcaPartInfo;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class MulcaClientFileDownloader implements FileDownloader<FileToDownloadViaMulcaClient> {
    private static final Logger logger = LoggerFactory.getLogger(MulcaClientFileDownloader.class);

    @Autowired
    private MulcaHttpClient mulcaHttpClient;
    @Autowired
    private MulcaClient mulcaClient;
    @Autowired
    private MimeDetector mimeDetector;

    @Value("${wmi.host}")
    private String wmiHost;
    @Value("${wmi.yt.host}")
    private String wmiYtHost;
    @Value("${mulca.namespace.mail}")
    private String mailMulcaNamespace;

    @Override
    public FileToDownloadViaMulcaClient buildFileToDownload(URI uri, StorageResourceInfo response) {
        MulcaId mulcaId = response.getStorageId().getMulcaId();
        boolean reparse = false;
        Option<String> encoding = Option.empty();
        Option<MulcaPartHeaders> partHeaders = Option.empty();

        if (!mulcaId.isOnlyStid()) {
            if (response.getStorageId().getMulcaRange().isPresent()) {
                encoding = response.getEncoding();
            } else {
                // Old attachments scheme
                checkMulcaIdPartForXpathInjection(mulcaId);

                MulcaPartInfo partInfo = mulcaClient.getMulcaPartInfo(mulcaId);
                partHeaders = Option.of(partInfo.partHeaders);
                encoding = Option.of(partHeaders.get().transferEncoding);
                reparse = partInfo.isPartFromRfc822;
            }
        }

        final Option<String> chosenContentDispositionFilename = response.getContentDispositionFileName()
                .orElse(partHeaders.flatMapO(headers -> headers.contentDispositionFileName))
                .orElse(partHeaders.flatMapO(headers -> headers.contentTypeName));

        final Option<String> chosenContentType = response.getContentType()
                .plus(partHeaders.map(headers -> headers.contentType))
                .plus(chosenContentDispositionFilename.map(mimeDetector.getMimeTypeByFilenameF()))
                .filter(MimeTypes.MIME_UNKNOWNS.containsF().notF())
                .firstO();
        boolean isArchive = MimeDetector.isArchive(chosenContentType.getOrElse(""));

        FileToDownloadViaMulcaClient fileToDownload = new FileToDownloadViaMulcaClient(
                uri, response.getStorageId(), chosenContentDispositionFilename,
                isArchive, reparse, encoding, chosenContentType);

        logger.debug("fileToDownload: {}", fileToDownload);

        return fileToDownload;
    }

    @Override
    public TempFileInfo download(FileToDownloadViaMulcaClient fileToDownload, MaxFileSizeChecker sizeChecker) {
        URI downloadUri = getDownloadUri(fileToDownload);

        HttpGet httpGet = new HttpGet(downloadUri);
        fileToDownload.getStorageId().getMulcaRange()
                .ifPresent(range -> httpGet.setHeader("Range", range.getHeaderValue()));

        logger.info("About to download file from {}, contentType: {}", downloadUri,
                fileToDownload.getContentType());
        return ApacheHttpClientUtils.execute(httpGet, mulcaHttpClient,
                new CopierResponseHandler(
                        downloadUri,
                        fileToDownload.getContentType(),
                        sizeChecker,
                        fileToDownload.getEncoding(),
                        mimeDetector))
                .withContentDispositionFilename(fileToDownload.getContentDispositionFilename())
                .withCacheResult(CacheResult.N_A);
    }


    private void checkMulcaIdPartForXpathInjection(MulcaId id) {
        if (!id.getPart().matches("[0-9.]*")) {
            throw new AssertionError(
                    "MulcaId part contains inappropriate symbols, mulca-id=" + id.toSerializedString());
        }
    }

    private URI getDownloadUri(FileToDownloadViaMulcaClient fileToDownloadViaMulcaClient) {
        StorageId storageId = fileToDownloadViaMulcaClient.getStorageId();
        // if range exists get only stid without part
        MulcaId realMulcaId = storageId.getMulcaRange().isPresent() ?
                              MulcaId.valueOf(storageId.getMulcaId().getStid(), "") : storageId.getMulcaId();

        String downloadUrl = mulcaClient.getDownloadUri(realMulcaId).toString();

        if (fileToDownloadViaMulcaClient.isReparse()) {
            downloadUrl = UrlUtils.addParameter(downloadUrl, "reparse", true);
        }
        if (StringUtils.isNotEmpty(mailMulcaNamespace) && isMailStorageRequest(fileToDownloadViaMulcaClient.getUri())) {
            downloadUrl = UrlUtils.updateParameter(downloadUrl, "ns", mailMulcaNamespace);
        }

        return UrlUtils.uri(downloadUrl);
    }

    private boolean isMailStorageRequest(URI uri) {
        return UriUtils.getHostO(uri).isMatch(host -> wmiHost.equals(host) || wmiYtHost.equals(host));
    }
}
