package ru.yandex.chemodan.app.docviewer.copy.provider;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author yashunsky
 */
public class TrackerUrlProvider extends DocumentUrlProvider {
    public static final String ORG_ID_PARAM = "org-id";
    public static final String UID_PARAM = "uid";

    private final String trackerApiHost;

    public TrackerUrlProvider(String trackerApiHost) {
        super("tracker");
        this.trackerApiHost = trackerApiHost;
    }

    @Override
    public String rewriteUrl(DocumentSourceInfo source) {
        TrackerDocumentId documentId = getDocumentId(source.getOriginalUrl());
        return UrlUtils.addParameter(
                getTrackerApiUrlBaseStartPart() + documentId.attachmentId + "/storageLink",
                ORG_ID_PARAM, documentId.orgId,
                UID_PARAM, source.getUid().getUid());
    }

    private TrackerDocumentId getDocumentId(String originalUrl) {
        String[] parts = StringUtils.substringAfter(originalUrl, getUrlPrefix()).split("/", 0);
        Validate.equals(2, parts.length);
        return new TrackerDocumentId(parts[0], parts[1]);
    }

    @NotNull
    private String getTrackerApiUrlBaseStartPart() {
        return "https://" + trackerApiHost + "/v2/attachments/";
    }

    @Override
    public Option<DiskResourceId> getDiskResourceId(ActualUri uri) {
        return Option.empty();
    }

    @Override
    public boolean isSupportedActualUri(ActualUri actualUri) {
        return StringUtils.startsWith(actualUri.getUriString(), getTrackerApiUrlBaseStartPart());
    }

    @Override
    public MapF<String, String> getSpecificHeaders() {
        return Cf.map();
    }

    @AllArgsConstructor
    @Data
    private static class TrackerDocumentId {
        private final String orgId;
        private final String attachmentId;
    }
}
