package ru.yandex.chemodan.app.docviewer.copy.resourcemanagers;

import java.net.URI;

import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.StorageResourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.storage.MdsStorageId;
import ru.yandex.chemodan.app.docviewer.copy.storage.StorageId;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.InternalHttpClient;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.MpfsHttpClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.mulca.MulcaRange;
import ru.yandex.inside.wmi.WmiAttachmentInfo;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author metal
 * TODO split by resourceManager which autowired by interface and auto matched by host
 */
public class StorageResourceInfoManager {
    @Autowired
    private InternalHttpClient internalHttpClient;
    @Autowired
    private MpfsHttpClient mpfsHttpClient;

    @Autowired
    @Qualifier("mailMetaHttpClient")
    private HttpClient mailMetaHttpClient;
    @Autowired
    @Qualifier("ytMailMetaHttpClient")
    private HttpClient ytMailMetaHttpClient;

    @Autowired
    private TrackerApiManager trackerApiManager;
    @Autowired
    private UslugiApiManager uslugiApiManager;
    @Autowired
    private SchoolbookApiManager schoolbookApiManager;
    @Autowired
    private PracticumApiManager practicumApiManager;

    @Value("${mpfs.host}")
    private String mpfsHost;

    @Value("${wmi.host}")
    private String wmiHost;
    @Value("${wmi.yt.host}")
    private String wmiYtHost;

    @Value("${mail_meta.host}")
    private String mailMetaHost;
    @Value("${yt_mail_meta.host}")
    private String ytMailMetaHost;
    @Value("${tracker_api.host}")
    private String trackerApiHost;
    @Value("${schoolbook_api.host}")
    private String schoolbookApiHost;
    @Value("${practicum_api.host}")
    private String practicumApiHost;

    @Value("${mulca.gate.host}")
    private String mulcaGateHost;
    @Value("${browser.mds.namespace}")
    private String namespace;
    @Value("${mds.host-port}")
    private HostPort mdsHostPort;

    public Option<StorageResourceInfo> getStorageResourceInfoResponse(URI uri) {
        String host = UriUtils.getHost(uri);
        if (mpfsHost.equals(host)) {
            Option<String> versionId = UrlUtils.getQueryParameterFromUrl(uri.toString(), "version_id");
            if (versionId.isPresent()) {
                return Option.of(getStorageResourceInfoResponseFromMpfsVersioning(uri));
            } else {
                return Option.of(getStorageResourceInfoResponseFromMpfs(uri));
            }
        } else if (mulcaGateHost.equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromMulca(uri));
        } else if (mdsHostPort.getHost().toString().equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromMds(uri));
        } else if (wmiHost.equals(host) || wmiYtHost.equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromMail2(uri));
        } else if (mailMetaHost.equals(host) || ytMailMetaHost.equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromMail3(uri));
        } else if (trackerApiHost.equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromRawUrl(trackerApiManager.getStorageUri(uri)));
        } else if (uslugiApiManager.isUrlMatched(uri.toString())) {
            return Option.of(getStorageResourceInfoResponseFromRawUrl(uslugiApiManager.getStorageUri(uri)));
        } else if (schoolbookApiHost.equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromRawUrl(schoolbookApiManager.getStorageUri(uri)));
        } else if (practicumApiHost.equals(host)) {
            return Option.of(getStorageResourceInfoResponseFromRawUrl(practicumApiManager.getStorageUri(uri)));
        }else {
            return Option.empty();
        }
    }

    private StorageResourceInfo getStorageResourceInfoResponseFromMulca(URI uri) {
        return new StorageResourceInfo(StorageId.fromMulcaId(MulcaId.fromHttpUri(uri)));
    }

    private StorageResourceInfo getStorageResourceInfoResponseFromMds(URI uri) {
        return new StorageResourceInfo(MdsStorageId.fromHttpUri(uri));
    }

    private StorageResourceInfo getStorageResourceInfoResponseFromRawUrl(URI uri) {
        return new StorageResourceInfo(StorageId.fromRawUrl(uri.toString()));
    }

    private StorageResourceInfo getStorageResourceInfoResponseFromMpfs(URI uri) {
        HttpGet httpGet = new HttpGet(uri);

        return ApacheHttpClientUtils
                .execute(httpGet, mpfsHttpClient, new YaDiskCopierResponseHandler(uri, TimeUtils.now()));
    }

    private StorageResourceInfo getStorageResourceInfoResponseFromMpfsVersioning(URI uri) {
        HttpGet httpGet = new HttpGet(uri);
        return ApacheHttpClientUtils
                .execute(httpGet, mpfsHttpClient, new YaDiskCopierResponseHandlerVersioning(uri, TimeUtils.now()));

    }

    private StorageResourceInfo getStorageResourceInfoResponseFromMail3(URI uri) {
        // part parameter is not used by wmi, we add it only to create unique url
        // for different attachments
        String url = uri.toString();
        String mid = UrlUtils.getQueryParameterFromUrl(url, "mid").getOrThrow("No part parameter");
        String hid = UrlUtils.getQueryParameterFromUrl(url, "part").getOrThrow("No part parameter");
        String realUrl = UrlUtils.removeParameter(uri.toString(), "part");

        String host = UriUtils.getHost(uri);
        HttpClient httpClient;
        if (mailMetaHost.equals(host)) {
            httpClient = this.mailMetaHttpClient;
        } else if (ytMailMetaHost.equals(host)) {
            httpClient = this.ytMailMetaHttpClient;
        } else {
            throw new IllegalStateException("Unknown host " + host);
        }

        WmiAttachmentInfo info = ApacheHttpClientUtils.execute(new HttpGet(realUrl), httpClient,
                new YaMailCopierResponseHandler2(UrlUtils.uri(url), mid, hid, Instant.now()));

        return new StorageResourceInfo(
                StorageId.fromMulcaId(
                        MulcaId.valueOf(info.stid, info.hid),
                        new MulcaRange(info.offsetBegin, info.offsetEnd)),
                info.name, info.getFullContentType(), info.encoding);
    }

    private StorageResourceInfo getStorageResourceInfoResponseFromMail2(URI uri) {
        // part parameter is not used by wmi, we add it only to create unique url
        // for different attachments
        String url = uri.toString();
        String mid = UrlUtils.getQueryParameterFromUrl(url, "mid").getOrThrow("No part parameter");
        String hid = UrlUtils.getQueryParameterFromUrl(url, "part").getOrThrow("No part parameter");
        String realUrl = UrlUtils.removeParameter(uri.toString(), "part");
        WmiAttachmentInfo info = ApacheHttpClientUtils.execute(new HttpGet(realUrl), internalHttpClient,
                new YaMailCopierResponseHandler2(UrlUtils.uri(url), mid, hid, Instant.now()));

        return new StorageResourceInfo(
                StorageId.fromMulcaId(
                        MulcaId.valueOf(info.stid, info.hid),
                        new MulcaRange(info.offsetBegin, info.offsetEnd)),
                info.name, info.getFullContentType(), info.encoding);
    }
}
